from typing import Optional, Union

import re
import attr
import logging

from staff.lib.requests import Session

from django.conf import settings
from django.db.models import ObjectDoesNotExist

from staff.person.models import Staff, StaffPhone, PHONE_KIND, PHONE_TYPES

logger = logging.getLogger('dialer')

PHONE_CLEAN_REGEX = re.compile(r'\D')
DIALER_SESSION = Session()


@attr.s(auto_attribs=True)
class DialerPhone:
    number: str
    owner_login: Optional[str] = None
    owner_name: Optional[str] = None


def format_name(login: str, first_name: str, last_name: str) -> str:
    name = f'{first_name} {last_name}'.strip()
    name = name or login
    return name


def clean_number(phone_number: Union[str, int]):
    phone = str(phone_number)
    phone = phone.strip()
    # Магические цифры ниже дали наши телефонисты
    if phone.startswith('+7'):
        phone = phone.replace('+7', '98')
    elif phone.startswith('+'):
        phone = phone.replace('+', '9810')
    elif len(phone) <= settings.MAX_WORK_PHONE_LEN:  # с рабочим не делаем ничего
        pass
    else:
        logger.warning('Calling on strange phone %s', phone)
    return re.sub(PHONE_CLEAN_REGEX, '', phone)


def get_person_work_phone(person: Staff) -> DialerPhone:
    return DialerPhone(
        number=clean_number(person.work_phone),
        owner_login=person.login,
        owner_name=format_name(person.login, person.first_name_en, person.last_name_en)
    )


def get_person_main_phone(login: str) -> Optional[DialerPhone]:
    phone = (
        StaffPhone.objects
        .values('number', 'staff__login', 'staff__first_name_en', 'staff__last_name_en')
        .filter(
            staff__login=login,
            staff__is_dismissed=False,
            intranet_status=1,
            kind__in=[PHONE_KIND.COMMON, PHONE_KIND.MONITORING],
        )
        .exclude(type=PHONE_TYPES.HOME)
        .order_by('position')
        .first()
    )
    if not phone:
        return None

    return DialerPhone(
        number=clean_number(phone['number']),
        owner_login=phone['staff__login'],
        owner_name=format_name(
            phone['staff__login'], phone['staff__first_name_en'], phone['staff__last_name_en']
        ),
    )


def get_phone_by_number(phone_number: str) -> DialerPhone:
    if len(phone_number) > settings.MAX_WORK_PHONE_LEN:
        phone = (
            StaffPhone.objects
            .values('staff__login', 'staff__first_name_en', 'staff__last_name_en')
            .filter(number=phone_number, staff__is_dismissed=False, intranet_status=1)
            .order_by('staff_id')
            .first()
        )
        if not phone:
            return DialerPhone(number=clean_number(phone_number))

        return DialerPhone(
            number=clean_number(phone_number),
            owner_login=phone['staff__login'],
            owner_name=format_name(phone['staff__login'], phone['staff__first_name_en'], phone['staff__last_name_en']),
        )

    try:
        person = (
            Staff.objects
            .values('login', 'first_name_en', 'last_name_en')
            .get(work_phone=phone_number, is_dismissed=False)
        )
        return DialerPhone(
            number=clean_number(phone_number),
            owner_login=person['login'],
            owner_name=format_name(person['login'], person['first_name_en'], person['last_name_en']),
        )
    except ObjectDoesNotExist:
        return DialerPhone(number=clean_number(phone_number))


class Dialer:
    DIALER_PROTOCOL = settings.DIALER_PROTOCOL
    DIALER_HOST = settings.DIALER_HOST
    DIALER_TIMEOUT = 1

    def __init__(self, call_from: DialerPhone, call_to: DialerPhone, from_: str = 'staff', backto: str = ''):
        self.call_from: DialerPhone = call_from
        self.call_to: DialerPhone = call_to
        self.from_: str = from_
        self.backto: str = backto

    @property
    def _params(self):
        params = {
            'calling': self.call_to.number,
            'caller': self.call_from.number,
            'callername': self.call_from.owner_name,
        }

        if self.from_:
            params['from'] = self.from_

        if self.backto:
            params['backto'] = self.backto

        if self.call_to.owner_login:
            params['callinglogin'] = self.call_to.owner_login

        return params

    @property
    def _url(self):
        return '{protocol}://{host}/cgi-bin/waycall.cgi'.format(
            protocol=self.DIALER_PROTOCOL,
            host=self.DIALER_HOST,
        )

    def call(self):
        try:
            answer = DIALER_SESSION.get(
                self._url, params=self._params, timeout=self.DIALER_TIMEOUT
            )
            if answer.status_code != 200:
                logger.error('Dialer returned an error %s', answer.status_code)
                return False
        except Exception:
            logger.exception('Dialer exception')
            return False
        return True
