from datetime import datetime, timedelta

from typing import List, Dict, Iterator, Any
from itertools import chain, groupby
from collections import defaultdict

from django.db.models import Q
from django.conf import settings
from django.http import HttpResponseBadRequest, HttpResponseForbidden
from django.views.decorators.http import require_GET
from django.utils.functional import SimpleLazyObject

from staff.departments.models import Department
from staff.lib import tvm2
from staff.lib.decorators import responding_json, available_for_external


from staff.person.models import (
    Staff,
    Contact,
    StaffPhone,
    PHONE_KIND,
    PHONE_PROTOCOLS,
)

from staff.lib.models.roles_chain import get_chiefs_by_persons
from staff.whistlah.utils import get_last_activity
from staff.gap.api.views.availability_views import get_availabilities
from staff.gap.controllers.utils import datetime_to_utc

EXT_TREE_ID = SimpleLazyObject(lambda: Department.get_tree_ids(['ext']))
ASSISTANT_CONTACT_TYPE_ID = 20
PERSON_FIELDS = (
    'id',
    'first_name', 'last_name', 'middle_name',
    'login',
    'position',
    'department_id', 'department__name',
    'office__name',
    'work_phone', 'work_email',
    'join_at',
    'is_dismissed',
)


def fetch_assistants_data(person_ids: List[int], observer_tvm_ticket: str, observer_uid: str) -> Dict[str, List]:
    """
    :param person_ids: список id сотрудиков, по которым нужно достать их ассистентов
    :return: assistants - данные об ассистентах для каждого person_id
    """
    # предполагаю, что у ассистент не может помогать больше чем одному человеку
    person_id_by_assistant_login = dict(Contact.objects.filter(
        person_id__in=person_ids,
        contact_type_id=ASSISTANT_CONTACT_TYPE_ID
    ).values_list('account_id', 'person_id'))

    assistants_data = (
        Staff.objects
        .filter(login__in=person_id_by_assistant_login.keys(), is_dismissed=False)
        .values(
            'login',
            'first_name',
            'last_name',
            'middle_name',
            'work_phone',
            'work_email',
        )
    )

    availabilities = get_persons_availabilities(assistants_data, observer_tvm_ticket, observer_uid)

    assistants = defaultdict(list)
    for assistant in assistants_data:
        assistant['availability'] = availabilities[assistant['login']]
        assistant.pop('work_email')
        assistants[person_id_by_assistant_login[assistant['login']]].append(assistant)

    return assistants


def get_persons_by_query(search_query: str, limit: int) -> List[Dict[str, Any]]:

    search_fields = ('first_name', 'first_name_en', 'last_name', 'last_name_en', 'middle_name', 'login')

    condition = Q()
    for search_item in search_query.split():
        part_of_condition = Q(work_phone__startswith=search_item) & Q(is_dismissed=False)
        for search_field in search_fields:
            part_of_condition |= Q(**{search_field + '__istartswith': search_item})
        condition &= part_of_condition

    persons = (
        Staff.objects
        .filter(is_robot=False)
        .filter(condition)
        .exclude(department__tree_id=EXT_TREE_ID['ext'])
        .extra({'work_phone': "CAST(work_phone as TEXT)"})
        .values(*PERSON_FIELDS)
        .order_by('is_dismissed')
    )[:limit]

    return persons


def add_phones(persons: List[Dict[str, Any]]) -> None:
    ids = [person['id'] for person in persons]
    ids.extend(
        cw['id'] for cw in chain.from_iterable(
            person['co-workers'] for person in persons
        )
    )

    mobile_phone_by_person_id = dict(
        StaffPhone.objects
        .filter(
            staff_id__in=ids,
            protocol=PHONE_PROTOCOLS.ALL,
            kind=PHONE_KIND.COMMON,
            intranet_status=1,
        )
        .order_by('-position')
        .values_list('staff', 'number')
    )

    for person in persons:
        for co_worker in person['co-workers']:
            if 'work_phone' not in co_worker or not co_worker['work_phone']:
                co_worker.pop('work_phone', None)
                co_worker['mobile_phone'] = mobile_phone_by_person_id.get(co_worker['id'])
        if 'work_phone' not in person or not person['work_phone']:
            person.pop('work_phone', None)
            person['mobile_phone'] = mobile_phone_by_person_id.get(person['id'])


def add_coworkers(persons: List[Dict[str, Any]]) -> None:
    co_workers = (
        Staff.objects
        .filter(is_dismissed=False, is_robot=False)
        .filter(department_id__in=[p['department_id'] for p in persons])
        .values(*PERSON_FIELDS)
        .order_by('department_id')
    )
    dep_persons = {}
    for dep_id, persons_list in groupby(co_workers, key=lambda cw: cw['department_id']):
        dep_persons[dep_id] = list(persons_list)

    for person in persons:
        dep_id = person['department_id']
        person['co-workers'] = []
        if not person['is_dismissed']:
            person['co-workers'] = [cw for cw in dep_persons[dep_id] if cw['login'] != person['login']]


def add_telegram_logins(persons):
    ids = [person['id'] for person in persons]
    ids.extend(chain.from_iterable([cw['id'] for cw in person['co-workers']] for person in persons))

    telegram_by_person_id = dict(
        Contact.objects
        .filter(person_id__in=ids, contact_type__name='Telegram')
        .order_by('-position')
        .values_list('person_id', 'account_id')
    )

    for person in persons:
        person['telegram'] = telegram_by_person_id.get(person['id'])
        for co_worker in person['co-workers']:
            co_worker['telegram'] = telegram_by_person_id.get(co_worker['id'])


def get_chiefs(persons: List[Dict[str, Any]]):
    persons_chiefs = get_chiefs_by_persons(
        [(person['id'], person['login'], person['department_id']) for person in persons]
    )

    chief_employees = defaultdict(list)
    for person in persons_chiefs:
        if person['chiefs']:
            chief_employees[person['chiefs'][0]['login']].append(person['person']['id'])

    chiefs = (
        Staff.objects
        .filter(login__in=chief_employees.keys())
        .values(
            'first_name',
            'last_name',
            'middle_name',
            'login',
            'work_phone',
        )
    )

    result = {}
    for chief in chiefs:
        for employee in chief_employees[chief['login']]:
            result[employee] = chief

    return result


def filter_person_fields(person):
    person_fields = [
        'first_name',
        'last_name',
        'middle_name',
        'position',
    ]

    if person['is_dismissed']:
        person_fields.append('is_dismissed')
    elif person['assistants']:
        person_fields.append('assistants')
    else:
        person_fields += [
            'login',
            'department',
            'join_at',
            'work_phone',
            'mobile_phone',
            'last_activity',
            'chief',
            'availability',
            'co-workers',
            'telegram',
            'office__name',
        ]

    co_worker_fields = [
        'first_name',
        'last_name',
        'middle_name',
        'login',
        'mobile_phone',
        'telegram',
        'work_phone',
        'position',
    ]

    person['co-workers'] = [
        {
            key: value
            for key, value in co_worker.items()
            if key in co_worker_fields
        }
        for co_worker in person['co-workers']
    ]

    return {
        key: value
        for key, value in person.items()
        if key in person_fields
    }


def get_persons_on_maternity(gaps, date_from):
    maternity = set()
    for gap in gaps:
        if gap['workflow'] == 'maternity' and gap['date_from'] <= date_from:
            maternity.add(gap['person_login'])
    return maternity


def get_persons_availabilities(
        persons: Iterator[Dict[str, Any]],
        observer_ticket: str,
        observer_uid: str = '',
) -> Dict[str, Dict[str, Any]]:

    date_from = datetime.utcnow()
    date_to = date_from + timedelta(days=10)

    logins = [person['login'] for person in persons]
    availabilities = get_availabilities(
        person_logins=logins,
        date_from=date_from,
        date_to=date_to,
        include_holidays=True,
        include_calendar=True,
        working_hour_from=None,
        working_hour_to=None,
        ignore_work_in_absence=False,
        observer_tvm_ticket=observer_ticket,
        observer_uid=observer_uid,
    )

    gaps = chain.from_iterable([availability.gaps for availability in availabilities.values()])
    persons_on_maternity = get_persons_on_maternity(gaps, date_from)

    availability_status = defaultdict(lambda: {'is_available': True, 'seconds_before': 0, 'on_maternity': False})

    for login, availability in availabilities.items():
        availability_status[login]['on_maternity'] = login in persons_on_maternity
        if availability.root:
            first_availability_date_from = datetime_to_utc(availability.root.date_from, availability.timezone)
            seconds_before = int((first_availability_date_from - date_from).total_seconds())
            availability_status[login]['seconds_before'] = seconds_before
            availability_status[login]['is_available'] = seconds_before == 0

    return availability_status


@available_for_external
@responding_json
@require_GET
def telephonist_search(request):
    search_query = request.GET.get('search_query', '').strip()
    limit = request.GET.get('limit', '20').strip()

    if not request.user.has_perm('api.has_access_to_callcenter'):
        return HttpResponseForbidden()

    if not search_query:
        return HttpResponseBadRequest('Empty search query')

    try:
        limit = int(limit)
    except ValueError:
        return HttpResponseBadRequest('Limit must be integer')

    persons = get_persons_by_query(search_query, limit)
    add_coworkers(persons)
    add_phones(persons)
    add_telegram_logins(persons)

    availabilities = get_persons_availabilities(
        persons,
        observer_ticket=tvm2.get_user_ticket_for_robot_staff(),
        observer_uid=settings.CALENDAR_USER_UID,  # robot-staff
    )
    chiefs = get_chiefs(persons)
    assistants = fetch_assistants_data(
        person_ids=[person['id'] for person in persons],
        observer_tvm_ticket=request.yauser.raw_user_ticket,
        observer_uid=settings.CALENDAR_USER_UID,  # robot-staff
    )
    for person in persons:
        person_id = person.pop('id')
        person['chief'] = chiefs.get(person_id, None)
        person['assistants'] = assistants.get(person_id, [])
        person['availability'] = availabilities[person['login']]
        person['department'] = person.pop('department__name')

    last_activities = get_last_activity([person['login'] for person in persons])
    for person in persons:
        activity = last_activities[person['login']]
        if activity:
            delta = (datetime.now() - activity['updated_at']).total_seconds()
            person['last_activity'] = {
                'office_name': activity['name'],
                'seconds_ago': int(delta),
            }

    persons = [filter_person_fields(person) for person in persons]

    return persons
