import logging
from datetime import date, datetime, timedelta
from io import BytesIO

import qrcode

from django.db.models import Q
from django.http import HttpResponseNotFound, HttpResponse
from django.utils.translation import ugettext as _
from django.views.decorators.csrf import csrf_exempt
from django.views.decorators.http import require_http_methods

from staff.lib import waffle
from staff.lib.decorators import (
    responding_jsonp,
    responding_json,
    make_json_response,
    available_by_center_token,
    available_by_tvm,
    available_for_external,
)
from staff.lib.models.roles_chain import get_chiefs_by_persons
from staff.lib.utils.qs_values import localize
from staff.person.controllers import PersonCtl, Person
from staff.person.models import Staff
from staff.person_profile.permissions.utils import attach_permissions_ctl
from staff.person_profile.permissions.check_permissions import can_view_phones
from staff.vcard.vcard_generator import _get_vcard, _get_vcards

from staff.api.dialer import Dialer, get_phone_by_number, get_person_main_phone, get_person_work_phone
from staff.api.forms import StaffChangeForm
from staff.api.tools import get_last_applied_proposal_info

logger = logging.getLogger(__name__)

DAY_OFF_WEEKDAY_SATURDAY = 6
DAY_OFF_WEEKDAY_SUNDAY = 0


class StaffApiGapException(Exception):
    pass


@attach_permissions_ctl
def get_vcard(request, login):
    """
    Получаем vCard пользователя
    yandex_login - разделённые через запятую логины
    @return: HttpResponse
    """
    requested_logins = login.split(',')
    if waffle.switch_is_active('rkn_mode'):
        staff_qs = Staff.objects.filter(login__in=requested_logins, is_dismissed=False)
    else:
        staff_qs = Staff.objects.filter(login__in=requested_logins)

    if len(staff_qs) == 1:
        person = staff_qs.first()
        include_contacts = can_view_phones(None, request.permissions_ctl.properties, person.login)
        vcard_stream = _get_vcard(person, include_contacts)
    else:
        vcard_stream = _get_vcards(staff_qs, can_view_phones(None, request.permissions_ctl.properties), extend=True)

    response = HttpResponse(vcard_stream, content_type='text/x-vcard')
    response['Content-Disposition'] = (
        'attachment; filename=%s.vcf' % _get_vcard_filename(login))
    return response


@attach_permissions_ctl
def get_qr(request, login):
    """
    Получаем vCard и редиректим на генератор QR-кода

    @return: HttpResponse
    """

    person = Staff.objects.get(login=login)

    if person.is_dismissed and waffle.switch_is_active('rkn_mode'):
        return HttpResponseNotFound()
    card = _get_vcard(person, can_view_phones(None, request.permissions_ctl.properties, login), extend=False)
    qr = qrcode.QRCode(
        version=1,
        error_correction=qrcode.constants.ERROR_CORRECT_L,
        box_size=5,
        border=2,
    )
    qr.add_data(card.strip())
    qr.make(fit=True)
    img = qr.make_image()

    stream = BytesIO()
    img.save(stream, 'PNG')
    stream.seek(0)
    data = stream.read()

    response = HttpResponse(data)
    response['Content-Type'] = 'image/png'
    response['Content-Length'] = len(data)

    return response


@responding_jsonp
def get_whistlah(request, yandex_login):
    """Получаем информацию об последнем месте и типе активности пользователя"""
    from staff.whistlah.utils import get_last_activity

    last_activity = get_last_activity(logins=[yandex_login])[yandex_login]

    if not last_activity:
        return {}

    last_activity = localize(last_activity)
    block_title = _('staff.blocks.activity.last-title')
    office_name = last_activity['name']
    humanized_timedelta = _humanize_timedelta(
        start=last_activity['updated_at']
    )

    return {
        'block': 'b-activity',
        'js': True,
        'content': [
            {
                'elem': 'weekly',
                'content': '',
            },
            {
                'elem': 'last',
                'content': '%s: %s (%s)' % (block_title, office_name,
                                            humanized_timedelta),
            },
        ],
    }


@responding_jsonp
def get_jabber_status(request, login_or_mail):
    return {
        'status': 'unavailable',
        'priority': "0",
        'status_tooltip': "Недоступен",
        'name': login_or_mail.split('@')[0],
        'text': ""
    }


def call(request, phone):
    """
    Звоним человеку на мобильный или рабочий

    @return: dict
    """
    if 'from' in request.POST:
        data = request.POST
    else:
        data = request.GET
    is_jsonp = data.get('format', 'jsonp') == 'jsonp'

    Dialer(
        call_from=get_person_work_phone(request.user.get_profile()),
        call_to=get_phone_by_number(phone),
        from_=data.get('from'),
        backto=data.get('backto'),
    ).call()

    return make_json_response(request, data={'response': 'ok'}, status=200, is_jsonp=is_jsonp)


@responding_json
@available_for_external
@require_http_methods(['GET'])
def call_main_to_main(request):
    """
    Звоним с основного номера человека на основной номер другого человека
    """
    called_login = request.GET.get('called_login')
    if not called_login:
        return {'success': False, 'errors': ['called_login_required']}, 400

    errors = []

    call_from = get_person_main_phone(request.user.username)
    if call_from is None:
        errors.append('caller_main_phone_not_found')

    call_to = get_person_main_phone(called_login)
    if call_to is None:
        errors.append('called_main_phone_not_found')

    if not errors and call_from.number == call_to.number:
        errors.append('identical_numbers')

    if not errors:
        result = Dialer(call_from=call_from, call_to=call_to).call()
        if not result:
            errors.append('call_start_failed')

    if errors:
        return {'success': False, 'errors': errors}, 400
    else:
        return {'success': True, 'errors': errors}, 200


def _humanize_timedelta(start, end=None):
    """Очеловечиваем множественные формы временной дельты"""

    from django.utils.translation import ungettext

    end = end or datetime.now()
    delta = end - start

    if delta.days > 0:
        return '%s %s %s' % (
            delta.days,
            ungettext('staff.blocks.activity.humanize.day', 'staff.blocks.activity.humanize.day', delta.days),
            _('staff.blocks.activity.humanize.ago'),
        )
    elif delta.seconds > 3600:
        hours = delta.seconds // (60 * 60)
        return '%s %s %s' % (
            hours,
            ungettext('staff.blocks.activity.humanize.hour', 'staff.blocks.activity.humanize.hour', hours),
            _('staff.blocks.activity.humanize.ago'),
        )
    else:
        minutes = delta.seconds // 60 + 1
        return '%s %s %s' % (
            minutes,
            ungettext('staff.blocks.activity.humanize.minute', 'staff.blocks.activity.humanize.minute', minutes),
            _('staff.blocks.activity.humanize.ago'),
        )


def _get_vcard_filename(name=None):
    if name and len(name) < 255:
        return name

    return 'vcards'


@responding_json
@csrf_exempt
@require_http_methods(['POST'])
@available_for_external('django_intranet_stuff.can_change_staff')
def change_staff(request, login):
    """Ручка меняющая стафа."""
    try:
        staff = Staff.objects.get(login=login)
    except Staff.DoesNotExist:
        msg = 'Person "%s" not found' % login
        logger.exception(msg)
        return {'success': False, 'error': msg}, 404

    form = StaffChangeForm(request.POST or request.GET)

    if form.is_valid():
        PersonCtl(staff).update(form.cleaned_data).save(request.user)

        extra = {field: form.cleaned_data.get(field)
                 for field in form.EXTRA_FIELDS}
        person = Person(staff)
        for field, value in extra.items():
            setattr(person, field, value)
        person.save()

        return {'success': True, 'error': False}
    else:
        errors = '; '.join('%s %s' % (k, v[0])
                           for k, v in form.errors.items())
        return {'success': False, 'error': errors}, 409


@require_http_methods(['POST'])
@csrf_exempt
@available_for_external('django_intranet_stuff.robot_can_update_services')
def update_plan_service(request, service_id):
    from staff.groups.service_update_tasks import ServiceUpdateTasks
    try:
        ServiceUpdateTasks.schedule_services_update([int(service_id)])
    except ValueError:
        logger.info('update_plan_service failed for %s', service_id, exc_info=True)
        return HttpResponse('', status=400)
    return HttpResponse('', status=202)


@responding_json
@available_by_center_token
def users_byod_access(request):
    login = request.GET.get('login')
    state = {'true': True, 'false': False}.get(request.GET.get('state'))
    lookup = {}

    if login:
        lookup.update(login=login)

    if state is not None:
        lookup.update(extra__byod_access=state)

    def warp(staff_dict):
        staff_dict['byod_access'] = staff_dict['extra__byod_access']
        del staff_dict['extra__byod_access']
        return staff_dict

    return [
        warp(s)
        for s in Staff.objects.filter(**lookup).values('login', 'extra__byod_access')
    ]


@available_by_tvm(['review'])
@responding_json
@require_http_methods(['GET'])
def staff_with_chiefs(request):
    half_a_year = timedelta(185)

    person_list = (
        Staff.objects
        .filter(Q(is_dismissed=False) | Q(quit_at__gt=date.today()-half_a_year))
        .filter(is_robot=False)
        .values_list('id', 'login', 'department_id')
    )

    data = {'chiefs': get_chiefs_by_persons(person_list)}
    data.update(get_last_applied_proposal_info())

    return data
