import re

from datetime import date, time, datetime

from django.db.models.query import QuerySet
from django.http import Http404

from staff.person.models import StaffPhone, StaffCar, Inflection
from staff.groups.models import Group, GroupResponsible

from staff.lib.decorators import available_by_center_token

from staff.apicenter.views import UserView
from staff.apicenter.utils import (
    get_fields,
    serialize,
    jsonize_datetime,
    convert_edu_status_backwards,
)
from staff.apicenter.v1.departments import DepartmentsView
from staff.apicenter.v1.groups import (
    GroupView,
    GroupsView,
    UserGroupsView,
)
from staff.lib.models.departments_chain import get_departments_tree
from staff.keys.models import SSHKey

# Users section

_default_user_fields = ('id', 'login', 'login_ld', 'first_name', 'last_name', 'work_email', 'work_phone', 'position')
_cases = ('accusative', 'prepositional', 'subjective', 'dative', 'genitive', 'ablative',)
_staff_related_models = {
    'ssh_keys': {
        'model': SSHKey,
        'fields': ('key', 'description'),
    },
    'phones': {
        'model': StaffPhone,
        'fields': ('number', 'type', 'description'),
    },
    'cars': {
        'model': StaffCar,
        'fields': ('model', 'plate'),
    }
}


@available_by_center_token
def user_groups(request, user, format='json'):
    return UserGroupsView()(request, user, format=format)


def _get_users(user_or_qs, fields):

    if not isinstance(user_or_qs, QuerySet):
        qs = UserView.get_qs(user_or_qs)
    else:
        qs = user_or_qs

    _fields = set(fields)
    _fields.add('login')

    _fields.difference_update((
        'office',
        'city',
        'department',
        'table',
        'floor',
        'photo_count',
        'departments',
        'inflections',
        'offce',
        'memorial_profile',
    ))

    if 'departments' in fields:
        departments_tree = get_departments_tree(
            list(qs.values_list('department_id', flat=True)),
            ('id', 'name', 'code', 'from_staff_id')
        )
        qs = qs.select_related('department')
        _fields.add('department_id')

    if 'office' in fields:
        qs = qs.select_related('office')
        _fields.update(('office__name',))

    if 'city' in fields:
        qs = qs.select_related('office__city')
        _fields.update((
            'office__id',
            'office__name',
            'office__from_staff_id',
            'office__city__id',
            'office__city__name',
        ))

    if 'department' in fields:
        qs = qs.select_related('department')
        _fields.update(('department__id', 'department__name', 'department__from_staff_id', 'department__code'))

    if 'table' in fields:
        qs = qs.select_related('table')
        _fields.update(('table__id', 'table__from_staff_id',))

    if 'floor' in fields:
        qs = qs.select_related('table__floor')
        _fields.update((
            'table__id',
            'table__from_staff_id',
            'table__floor__id',
            'table__floor__name',
            'table__floor__from_staff_id',
        ))
    if 'inflections' in fields:
        _fields.update(('first_name', 'last_name'))
    if 'edu_status' in fields:
        _fields.update(('edu_direction',))

    if 'memorial_profile' in fields:
        qs = qs.select_related('memorial_profile')
        _fields.update((
            'memorial_profile__death_date',
            'memorial_profile__status_block_html',
            'memorial_profile__intranet_status',
        ))

    def convert_to_unicode(source_dict):
        for k, v in source_dict.items():
            if k == 'department_id':
                continue

            if isinstance(v, datetime):
                v = v.replace(microsecond=0)

            source_dict[k] = str(v) if v is not None else ''
        return source_dict

    def convert_to_tree(source_dict):
        result = {}
        for k, v in source_dict.items():
            key_list = k.split('__')
            prev_level = result
            len_key_list = len(key_list)
            for i in range(len_key_list):
                if i == len_key_list - 1:
                    prev_level[key_list[i]] = v
                else:
                    if key_list[i] not in prev_level:
                        prev_level[key_list[i]] = {}
                    prev_level = prev_level[key_list[i]]
        return result

    def remove_login(source_dict):
        if 'login' not in fields:
            del source_dict['login']
        return source_dict

    def set_departments(source_dict):
        if 'departments' in fields:
            departments = departments_tree[source_dict['department_id']]
            source_dict['departments'] = departments
            del source_dict['department_id']
        return source_dict

    def set_inflections(source_dict):
        if 'inflections' in fields:
            name = ('%s %s' % (source_dict['first_name'],
                               source_dict['last_name'])).strip()
            name = name or source_dict['login']

            if source_dict['login'] in Inflection.exceptions:
                exception = Inflection.exceptions[source_dict['login']]
                inflections = dict((_cases[i], exception[i+1]) for i in range(6))
                source_dict['inflections'] = inflections
            else:
                inflections = Inflection(name)
                source_dict['inflections'] = dict(zip(_cases, inflections))

            del source_dict['first_name']
            del source_dict['last_name']
        return source_dict

    def set_office(source_dict):
        if 'office' in fields:
            source_dict['office'] = source_dict['office']['name']
        return source_dict

    def convert_edu_status(source_dict):
        if 'edu_status' in fields:
            source_dict['edu_status'] = convert_edu_status_backwards(
                edu_status=source_dict['edu_status'],
                edu_direction=source_dict['edu_direction'],
            )
        if 'edu_direction' in source_dict and 'edu_direction' not in fields:
            del source_dict['edu_direction']
        return source_dict

    def remove_empty_memorials(source_dict):
        if 'memorial_profile' in fields:
            if source_dict['memorial_profile']['intranet_status'] != '1':
                source_dict['memorial_profile'] = None
            else:
                del source_dict['memorial_profile']['intranet_status']
        return source_dict

    qs = qs.values(*_fields)

    def apply_modifications(u):
        result = convert_to_unicode(u)
        result = convert_to_tree(result)
        result = convert_edu_status(result)
        result = set_office(result)
        result = set_departments(result)
        result = set_inflections(result)
        result = remove_login(result)
        result = remove_empty_memorials(result)
        return result

    result = dict(
        (u['login'], apply_modifications(u))
        for u in qs.iterator()
    )

    return result


@available_by_center_token
def user(request, user, format='json'):
    users_data = _get_users(user, get_fields(request, _default_user_fields))
    if not users_data:
        raise Http404()
    if len(users_data) == 1:
        users_data = list(users_data.values())[0]
    jsonize_datetime(users_data, ('modified_at', 'created_at'))
    return serialize(request, users_data, format, ('user', ))


@available_by_center_token
def current_user(request, format='json'):
    if not (request.user and request.user.is_authenticated()):
        users_data = {}
    else:
        users_data = _get_users(request.user.username, get_fields(request, _default_user_fields))
        users_data = list(users_data.values())[0]
    return serialize(request, users_data, format, ('user',))


# Groups section

_default_group_fields = ('id', 'name', 'code', 'url', 'description')


def _get_group(group):
    try:
        return Group.tree.get(url=group)
    except Group.DoesNotExist:
        try:
            return Group.tree.get(id=int(group))
        except (Group.DoesNotExist, ValueError):
            return False


def _get_group_all_members(group, requested_fields=None, include_dismissed=False):
    if not group:
        return []
    g = _get_group(group)
    if not g:
        return []

    members = g.get_all_members(include_dismissed=include_dismissed)

    if requested_fields:
        fields = [
            'department__url' if f == 'is_external_employee' else f
            for f in requested_fields
        ]
        members = members.values(*fields)

    def to_unicode(person):
        result = {}
        for field, value in person.items():
            if isinstance(value, (datetime, time, date)):
                value = re.sub(r'\.\d{6}', '', value.isoformat())
            if value is None:
                value = ''
            if not field.endswith('_id'):
                value = str(value)

            if field == 'department__url':
                if 'is_external_employee' in requested_fields:
                    result['is_external_employee'] = not value.startswith('yandex')
                if 'department__url' not in requested_fields:
                    continue

            result[field] = value
        return result

    return [to_unicode(m) for m in members.iterator()]


@available_by_center_token
def group_all_members(request, group, format='json'):
    include_dismissed = request.GET.get('include_dismissed', request.POST.get('include_dismissed', False))
    if include_dismissed in ('no', '0', 'false', False):
        include_dismissed = False
    else:
        include_dismissed = True

    if _check_option_group(request, group):
        r = []
    else:
        r = _get_group_all_members(
            group,
            get_fields(request, _default_user_fields),
            include_dismissed=include_dismissed,
        )
    return serialize(request, r, format, ('users', 'user'))


def _get_group_members(group, fields=None, include_dismissed=False):
    if not group:
        return []
    g = _get_group(group)
    if not g:
        return []
    # return serialize_models_list(g.members.filter(is_dismissed=False), None, fields)
    members = g.members.order_by('last_name', 'first_name', 'login')
    if not include_dismissed:
        members = members.filter(is_dismissed=False)

    users_data = _get_users(members, fields)
    users_data = list(users_data.values())
    return users_data


def _check_option_group(request, group_url_or_id):
    if group_url_or_id in ('option', str(Group.OPTION)):
        filter = {'staff__login': request.user.username}
        if group_url_or_id == 'option':
            filter['group__url'] = group_url_or_id
        else:
            filter['group__id'] = group_url_or_id
        if not request.user.is_authenticated() or GroupResponsible.objects.filter(**filter).count() == 0:
            return True
    return False


@available_by_center_token
def group(request, id, format='json'):
    return GroupView()(request, id, format=format)


@available_by_center_token
def groups(request, format='json'):
    return GroupsView()(request, format=format)


# Departments section

@available_by_center_token
def departments(request, format='json'):
    return DepartmentsView()(request, format=format)
