from django.db.models import Q
from django.http import Http404, HttpResponse
from json import dumps

from staff.departments.models import Department
from staff.lib.models.base import get_i_field

from staff.django_api_auth.decorators import check_referer

from staff.apicenter.autocomplete.utils import fetch_dict
from staff.apicenter.utils import get_fields


class DepartmentsAutocomplete(object):

    _default_fields = ('id', 'name', 'name_en', 'code', 'native_lang',
                       'tree_id', 'lft', 'rght')
    _ancestor_fields = ('id', 'name', 'name_en', 'code', 'native_lang')

    @check_referer
    def __call__(self, request):
        req_data = request.GET or request.POST
        q = req_data.get('q', None)
        if not q:
            raise Http404
        try:
            limit = int(req_data['limit'])
        except (TypeError, KeyError):
            limit = 10
        fields = get_fields(request, self._default_fields)

        qs = (
            Department.objects
            .values(*fields)
            .filter(
                Q(name__icontains=q) |
                Q(code__icontains=q) |
                Q(name_en__icontains=q)
            )
            .filter(intranet_status__gt=0)
        )

        r = []
        from django.db import connections
        cursor = connections['default'].cursor()
        sql = ('SELECT %s FROM intranet_department' +
               ' WHERE tree_id=%%d AND lft<%%d AND rght>%%d' +
               ' ORDER BY lft ASC') % (','.join(self._ancestor_fields), )
        for dep in qs.order_by('level')[:limit]:
            dep['name'] = get_i_field(dep, 'name')
            cursor.execute(sql % (dep['tree_id'], dep['lft'], dep['rght']))
            dep['ancestors'] = fetch_dict(cursor)
            for anc in dep['ancestors']:
                anc['name'] = get_i_field(anc, 'name')
                del anc['name_en'], anc['native_lang']
            del dep['name_en'], dep['native_lang'], dep['tree_id'], dep['lft'], dep['rght']
            r.append(dep)
        r = dumps(r)

        if req_data.get('callback', None):
            r = 'var data = %s; %s(data);' % (r, req_data.get('callback'))
        return HttpResponse(r, content_type='text/javascript')
