import re

from django.db.models import Q
from django.http import Http404, HttpResponse
from django.utils.translation import get_language

from staff.lib.decorators import available_by_center_token
from staff.person.models import Staff
from staff.lib.models.base import get_i_field

from staff.lib.json import dumps

from staff.apicenter.views import BaseCenterView


@available_by_center_token
class AutocompleteUsersView(BaseCenterView):

    fields = (
        'id',
        'login_ld',
        'login',
        'native_lang',
        'first_name',
        'last_name',
        'first_name_en',
        'last_name_en',
        'middle_name',
        'work_phone',
        'department',
        'position',
        'work_email',
    )
    staff_fields = [f.name for f in Staff._meta.fields]

    def __call__(self, request):
        users = self.get_users(request)
        res = dumps(users)
        callback = request.GET.get('callback', request.POST.get('callback', None))
        if callback:
            res = '%s(%s);' % (callback, res)
        return HttpResponse(res, content_type='text/javascript')

    def get_users(self, request):
        if 'q' in request.POST:
            data = request.POST
        else:
            data = request.GET

        q = data.get('q', '')
        try:
            limit = int(data['limit'])
        except (TypeError, KeyError):
            limit = 50
        if 'fields' in data:
            fields = [f for f in data.get('fields').split('|') if f in self.staff_fields]
            for f in self.fields:
                if f not in fields:
                    fields.append(f)
        else:
            fields = list(self.fields)
        try:
            is_dismissed = bool(int(data.get('is_dismissed', False)))
        except ValueError:
            is_dismissed = False

        if not q:
            raise Http404

        chunks = q.split(' ')
        chunks_len = len(chunks)

        # login@domain
        if q.find('@') > 0:
            query = Q(work_email__istartswith=q) | Q(home_email__istartswith=q)

        # name surname
        elif chunks_len == 2:
            if re.search('^[a-zA-Z]+$', chunks[0]):
                query = Q(en_name__icontains=chunks[0]) & Q(en_name__icontains=chunks[1])
            else:
                query = (Q(first_name__istartswith=chunks[0]) & Q(last_name__istartswith=chunks[1])) | \
                        (Q(first_name__istartswith=chunks[1]) & Q(last_name__istartswith=chunks[0]))

        # office phone number
        elif re.search(r'^\d+$', q):
            query = Q(work_phone__startswith=q)

        # mobile phone number
        elif re.search(r'^[\d\(\)-\+ ]+$', q):
            query = Q(mobile_phone_idx__contains=re.sub(r'[\(\)\-\+ ]', '', q))

        # simple search in login, name, surname, en_name
        else:
            if re.search(r'^[a-zA-Z0-9\.\-]+$', q):
                query = Q(login__istartswith=q) | Q(login_passport__istartswith=q) | Q(en_name__icontains=q)
            else:
                query = Q(first_name__istartswith=q) | Q(last_name__istartswith=q)

        users = Staff.objects.filter(query).exclude(login__isnull=True)
        if not is_dismissed:
            users = users.filter(is_dismissed=is_dismissed)
        lang = get_language()
        if lang == 'en':
            users = users.order_by('last_name_en', 'first_name_en', 'login')
        else:
            users = users.order_by('last_name', 'first_name', 'login')

        if 'department' in fields:
            is_department = True
            fields.extend(('department__name', 'department__name_en', 'department__native_lang'))
            users = users.select_related('department')
        else:
            is_department = False

        users = list(users.values(*fields)[0:limit])

        for u in users:
            q_exact_equals_name = u['first_name']+' '+u['last_name'] == q
            q_exact_equals_en_name = u['first_name_en']+' '+u['last_name_en'] == q
            if u['login'] == q or (q_exact_equals_name or q_exact_equals_en_name):
                # CENTER-112
                users.remove(u)
                users.insert(0, u)
            if is_department:
                u['department'] = get_i_field(u, 'name', 'department__')
                del u['department__name'], u['department__name_en'], u['department__native_lang']
            u['first_name'] = get_i_field(u, 'first_name')
            u['last_name'] = get_i_field(u, 'last_name')
            del u['first_name_en'], u['last_name_en'], u['native_lang']
        return users
