from django.conf import settings
from django.core.cache import cache
from django.core.exceptions import ValidationError

from staff.django_rpc import RpcException
from staff.django_rpc.decorators import rpc_method, AccessDenied

from staff.dislog.models import Log
from staff.lib.requests import Session
from staff.racktables.tasks import get_nets_by_macro
from staff.racktables.objects import YaIP

_models = (
    'City',
    'Country',
    'Department',
    'DepartmentKind',
    'DepartmentResponsible',
    'DepartmentStaff',
    'DepartmentStaffCounter',
    'Floor',
    'Group',
    'GroupMembership',
    'GroupPermission',
    'GroupResponsible',
    'MemorialProfile',
    'Office',
    'Organization',
    'Room',
    'Staff',
    'StaffCar',
    'StaffPhone',
    'Table',
    'TableReserve',
    'TableBook',
)

puncher_session = Session()
puncher_session.headers.update({
    'Authorization': 'OAuth {token}'.format(token=settings.ROBOT_STAFF_PUNCHER_OAUTH_TOKEN),
    'Content-type': 'application/json',
})
PUNCHER_RULES_URL = 'https://api.puncher.yandex-team.ru/api/dynfw/rules'
CACHE_NAME = 'update_center_ip_cache'
ALLOWED_NETWORKS = {
    '_MAILNETS_',
    '_LIBRA_TEST_NETS_',
    '_LIBRA_PROD_NETS_',
    '_STAFF_PROD_NETS_',
    '_STAFF_TEST_NETS_',
}


class IncorrectDateFormat(RpcException):
    pass


def get_allowed_networks_rules():
    key = 'xmlrpc_allowed_nets'
    networks = cache.get(key)

    if not networks:
        networks = [
            net.as_rule()
            for macro in ALLOWED_NETWORKS
            for net in get_nets_by_macro(macro)
        ]
        cache.set(key, networks, 7200)

    return networks


def check_network(ip):
    ip_rule = YaIP(ip).as_rule()
    rules = get_allowed_networks_rules()
    return any([ip_rule.is_inside(rule) for rule in rules])


@rpc_method('getUpdates', module='api')
def get_updates(modified_at=None, models=_models, **kwargs):
    """
    Getting updates for django_intranet_stuff app

    @param modified_at: datetime ('%Y-%m-%d %H:%M:%S') or ('%Y-%m-%d %H:%M:%S')
            of last update in your django_intranet_stuff app
    @type modified_at: str

    @param models: list of models your django_intranet_stuff app needs to update (by default all except groups models)
    @type models: list or tuple

    @return: list of dicts formed from Log model (meta information about all updated models)
    @rtype: list of dicts
    """

    if not check_network(kwargs['request'].real_user_ip):
        raise AccessDenied('Access denied')

    qs = Log.objects.filter(model_name__in=models)

    if modified_at:
        q = {'modified_at__gte': modified_at}
        try:
            qs = qs.filter(**q)
        except ValidationError:
            raise IncorrectDateFormat(modified_at)

    r = qs.order_by('modified_at')

    def modify_item(item):
        item['id'] = str(item['id'])
        item['modified_at'] = item['modified_at'].strftime('%Y-%m-%d %H:%M:%S')
        return item

    return [modify_item(item) for item in r.values()]
