from io import BytesIO
from xml.etree.cElementTree import Element, ElementTree

from django.http import HttpResponse
from django.utils.translation import gettext

from staff.lib.json import dumps

from staff.person.models import EDU_STATUS, EDU_DIRECTION


_DATETIME_FORMATS = (
    '%Y-%m-%d %H:%M:%S',
    '%Y-%m-%d %H:%M',
    '%Y-%m-%dT%H:%M:%S',
    '%Y-%m-%dT%H:%M',
    '%Y-%m-%d',
)


def xml_serialize(data, tags=None):
    if not tags:
        tags = tuple()
    level = 0
    content_elem = Element(_get_tag(tags, level))
    _xml_serialize(content_elem, data, tags, level)
    tree = ElementTree(content_elem)
    f = BytesIO()
    tree.write(f, 'UTF-8', xml_declaration=True)
    return f.getvalue()


def _get_tag(tags, level):
    if len(tags) > level:
        return tags[level]
    else:
        if level:
            return 'item'
        else:
            return 'content'


def _xml_serialize(parent_elem, data, tags, level):
    level += 1
    if isinstance(data, list) or isinstance(data, tuple):
        _xml_serialize_list(parent_elem, data, tags, level)
    elif isinstance(data, dict):
        _xml_serialize_dict(parent_elem, data, tags, level)
    elif isinstance(data, bool) or data is None:
        parent_elem.text = '1' if data else ''
    else:
        parent_elem.text = str(data)


def _xml_serialize_list(parent_elem, data_list, tags, level):
    for i in data_list:
        item_elem = Element(_get_tag(tags, level))
        parent_elem.append(item_elem)
        _xml_serialize(item_elem, i, tags, level)


def _xml_serialize_dict(parent_elem, data_dict, tags, level):
    for k, v in data_dict.items():
        key_elem = Element(k)
        parent_elem.append(key_elem)
        _xml_serialize(key_elem, v, tags, level)


def get_fields(request, default=None, var_name='fields', model=None):
    fields = request.GET.get(var_name, request.POST.get(var_name, None))
    if model and fields == '__all__':
        from staff.groups.models import Group
        if model == Group:
            return [f.name for f in model._meta.fields if f.name != 'name_en']
        return [f.name for f in model._meta.fields]
    if fields:
        return fields.split('|')
    return default


def serialize(request, data, _format, xml_tags=None):
    if not xml_tags:
        xml_tags = tuple

    if _format:
        _format = _format.strip('.')

    if _format == 'xml':
        return HttpResponse(xml_serialize(data, xml_tags), content_type='text/xml')

    cb = request.GET.get('callback', request.POST.get('callback', None))
    if _format == 'jsonp' or cb:
        return HttpResponse(
            'var data = %s; %s(data);' % (dumps(data), cb),
            content_type='text/javascript'
        )

    return HttpResponse(dumps(data), content_type='text/javascript')


def jsonize_datetime(data, datetime_fields):
    if not datetime_fields:
        return
    if type(data) == tuple:
        data = list(data)
    if type(data) == list:
        for i in range(len(data)):
            jsonize_datetime(data[i], datetime_fields)
    elif type(data) == dict:
        for i in data:
            if type(data[i]) == str and i in datetime_fields:
                data[i] = data[i].replace(' ', 'T')
            elif type(data[i]) == dict:
                jsonize_datetime(data[i], datetime_fields)


def convert_edu_status_backwards(edu_status='', edu_direction=''):
    """Преобразуем значения образования для обратной совместимости"""
    if edu_status in (EDU_STATUS.SPECIALIST,
                      EDU_STATUS.BACHELOR,
                      EDU_STATUS.MASTER):
        if edu_direction in (EDU_DIRECTION.TECHNICAL,
                             EDU_DIRECTION.LIBERAL,
                             EDU_DIRECTION.NATURAL,
                             EDU_DIRECTION.ECONOMIC):
            return edu_direction
        else:
            return ''
    else:
        return edu_status


# stupid gettext cannot understand "gt" func as gettext
gettext('api.date_ago_Just')
gettext('api.date_ago_xMinuteAgo')
gettext('api.date_ago_xMinutesAgo')
gettext('api.date_ago_MoreThanHourAgo')
gettext('api.date_ago_xHourAgo')
gettext('api.date_ago_xHoursAgo')
gettext('api.date_ago_Yesterday')
gettext('api.date_ago_DayBefore')
gettext('api.date_ago_DayAgo')
gettext('api.date_ago_DaysAgo')
gettext('api.date_ago_AgesAgo')
