import sys
import inspect

from django.http import Http404, HttpResponse
from django.contrib.auth.models import Permission

from staff.groups.models import GroupMembership
from staff.lib.decorators import responding_json, require_permission
from staff.users.models import User


can_view_permissions = require_permission('django_intranet_stuff.can_view_permissions', exception=Http404)


def get_name(p):
    return '{p.first_name} {p.middle_name} {p.last_name}'.format(p=p)


@can_view_permissions
def inspect_code(request):
    return HttpResponse(
        inspect.getsource(sys.modules[__name__]),
        content_type='text/plain'
    )


@can_view_permissions
@responding_json
def super_users(request):
    users = (
        User.objects
        .filter(is_superuser=True, staff__is_dismissed=False)
        .select_related('staff__department')
    )
    return [
        {
            'login': user.staff.login,
            'name': get_name(user.staff),
            'department_url': user.staff.department.url,
            'department_name': user.staff.department.name,
        }
        for user in users
    ]


@can_view_permissions
@responding_json
def holders_permissions(request):
    requested_perms = request.GET.get('perms')
    if requested_perms:
        requested_perms = requested_perms.split(',')
    else:
        requested_perms = [
            'can_edit_department',
            'can_edit_head',
            'can_edit_keys',
            'kill_and_rob',
            'can_dismiss_from_anketa',
            'can_dismiss_from_chit',
            'add_department',
            'change_department',
            'delete_department',
            'add_staff',
            'change_staff',
            'delete_staff',
            'add_departmentstaff',
            'change_departmentstaff',
            'delete_departmentstaff',
        ]

    def permission_as_dict(person, permission, date=''):
        return {
            'login': person.login,
            'name': get_name(person),
            'permission': permission.codename,
            'permission_name': permission.name,
            'department_url': person.department.url,
            'department_name': person.department.name,
            'date': date,
        }

    granted_perms = Permission.objects.filter(codename__in=requested_perms)

    result = []

    for permission in granted_perms:
        users = (
            permission.user_set
            .filter(staff__is_dismissed=False)
            .select_related('staff__department')
        )
        for user in users:
            result.append(permission_as_dict(user.staff, permission))

        for group in permission.groups.all():
            members = (
                GroupMembership.objects
                .filter(group=group, staff__is_dismissed=False)
                .select_related('staff__department')
            )
            for member in members:
                result.append(permission_as_dict(
                    member.staff, permission, member.joined_at
                ))

    return result
