from difflib import Differ

from django.shortcuts import render
from django.http import Http404

import json

from staff.lib.paginator import paginate
from staff.lib.decorators import require_permission
from staff.audit.models import Log
from staff.audit.factory import RENDER_AS_IS
from staff.audit.forms import AuditFilter


def _to_string(s):
    if not s:
        return ''
    return json.dumps(
        json.loads(s.data),
        indent=2,
        ensure_ascii=False,
    )


def filter_logs(form, log_qs):
    fields_map = [
        ('who', 'who__username'),
        ('action', 'action__contains'),
        ('primary_key', 'primary_key'),
        ('modified_from', 'modified_at__gte'),
        ('modified_to', 'modified_at__lt'),
    ]

    for form_field, filter_op in fields_map:
        value = form.cleaned_data[form_field]
        if value:
            filter_kwargs = {filter_op: value}
            log_qs = log_qs.filter(**filter_kwargs)

    return log_qs


@require_permission('audit.can_see_audit', exception=Http404)
def index(request):
    form = AuditFilter(request.GET or None)

    logs = Log.objects.select_related('who').order_by('-modified_at')
    if form.is_valid():
        logs = filter_logs(form, logs)

    paginated_logs = paginate(object_list=logs, per_page=50,
                              page=request.GET.get('page'))

    filter_query = request.GET.copy()
    filter_query.pop('page', None)
    return render(request, 'audit/index.html', {
        'form': form,
        'paginated_logs': paginated_logs,
        'filter_query': filter_query
    })


@require_permission('audit.can_see_audit', exception=Http404)
def blob(request, pk):
    viewing_log = Log.objects.get(pk=pk)
    prev_log = Log.objects.filter(
        action=viewing_log.action,
        primary_key=viewing_log.primary_key,
        modified_at__lte=viewing_log.modified_at,
        id__lt=viewing_log.id,
    ).order_by('-modified_at', "-id").first()
    next_log = Log.objects.filter(
        action=viewing_log.action,
        primary_key=viewing_log.primary_key,
        modified_at__gte=viewing_log.modified_at,
        id__gt=viewing_log.id,
    ).order_by('modified_at', 'id').first()

    if viewing_log.action in RENDER_AS_IS:
        data = _to_string(viewing_log)
    else:
        data = '\n'.join(Differ().compare(
            _to_string(prev_log).split('\n'),
            _to_string(viewing_log).split('\n'),
        ))

    return render(request, 'audit/blob.html', {
        'prev_log': prev_log,
        'viewing_log': viewing_log,
        'next_log': next_log,
        'data': data,
    })
