import logging

from typing import Optional

from functools import wraps

from django import forms
from django.conf import settings
from django.contrib import admin, messages
from django.contrib.admin.templatetags.admin_urls import add_preserved_filters
from django.core.exceptions import ValidationError
from django.db.models import Q
from django.http import HttpResponseRedirect

from staff.departments.models import Department, InstanceClass
from staff.lib.db import atomic
from staff.lib.utils.admin import BaseIntranetAdmin, AutocompleteMixin
from staff.person.models import Occupation

from staff.budget_position.models import (
    ChangeRegistry,
    GeographyMapping,
    OEBSGrade,
    PersonSchemeExceptions,
    ProfLevelMapping,
    Review,
    Reward,
    Workflow,
)
from staff.budget_position.const import WORKFLOW_STATUS, PUSH_STATUS
from staff.budget_position.workflow_service import (
    WorkflowRegistryService,
    OEBSError,
    use_cases,
    gateways,
    entities,
)


logger = logging.getLogger(__name__)


def describe(description, admin_order_field=None):
    def decorator(func):
        func.short_description = description
        if admin_order_field:
            func.admin_order_field = admin_order_field
        return func
    return decorator


CAN_CONFIRM_CREDIT_MANAGEMENT_CODES = (
    entities.workflows.WorkflowCreditManagement.code,
    entities.workflows.WorkflowCreditManagementWithVacancy.code,
)


class TemplateForWorkflowRegistry:
    change_form_template = 'workflow_change_form.html'

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.workflow_service = WorkflowRegistryService()
        self._get_oebs_link_usecase = use_cases.GetOEBSLink(
            gateways.OEBSService(gateways.BudgetPositionsRepository()),
            gateways.WorkflowRepository(),
        )

    def change_view(self, request, object_id, form_url='', extra_context=None):
        extra_context = extra_context or {}
        wf = Workflow.objects.get(id=object_id)
        extra_context['can_confirm_credit_management'] = (
            wf.code in CAN_CONFIRM_CREDIT_MANAGEMENT_CODES
            and wf.status == WORKFLOW_STATUS.PENDING
        )
        extra_context['can_push_workflow'] = wf.status == WORKFLOW_STATUS.CONFIRMED
        extra_context['can_cancel'] = self.workflow_service.workflow_can_be_cancelled(object_id)
        extra_context['can_resend_failed_to_oebs'] = (
            wf.status == WORKFLOW_STATUS.FINISHED
            and wf.changeregistry_set.filter(push_status=PUSH_STATUS.ERROR).exists()
        )

        return super().change_view(request, object_id, form_url, extra_context)

    def _mark_as_processed_manually(self, request, obj, redirect_url, _):
        return self._wrap_errors(
            lambda: self.workflow_service.mark_workflow_processed_manually(obj.id, request.user.get_profile().id),
            'Принято.',
            'Ошибка при маркировке',
            request,
            redirect_url,
        )

    def _send_to_oebs(self, request, obj, redirect_url, _):
        return self._wrap_errors(
            lambda: self.workflow_service.push_workflow_to_oebs(obj.id, request.user.get_profile().id),
            'Отправлено в OEBS.',
            'Ошибка при отправке в OEBS',
            request,
            redirect_url,
        )

    def _resend_failed_to_oebs(self, request, obj, redirect_url, _):
        return self._wrap_errors(
            lambda: self.workflow_service.push_failed_workflows_to_oebs([obj.id], request.user.get_profile().id),
            'В очередеи на отправку в OEBS.',
            'Ошибка при отправке в OEBS',
            request,
            redirect_url,
        )

    def _confirm_workflow(self, request, obj, redirect_url, _):
        return self._wrap_errors(
            lambda: self.workflow_service.confirm_workflow(obj.id, request.user.get_profile().id),
            'Успешно согласовано и отправлено в OEBS.',
            'Ошибка при согласовании',
            request,
            redirect_url,
        )

    def _wrap_errors(self, call, success_message, fail_message, request, redirect_url):
        try:
            call()
            self.message_user(request, success_message, messages.SUCCESS)
        except OEBSError as e:
            logger.info('Exception during operation from admin', exc_info=True)
            self.message_user(request, f'{fail_message}: {str(e)}', messages.ERROR)
        except Exception as e:
            logger.exception('Exception during operation from admin')
            self.message_user(request, f'{fail_message}: {str(e)}', messages.ERROR)

        return HttpResponseRedirect(redirect_url)

    def _cancel(self, request, obj, redirect_url, _):
        self.workflow_service.cancel_workflow(obj.id, request.user.get_profile().id)
        return HttpResponseRedirect(redirect_url)

    def _view_position(self, request, obj, redirect_url, action_name):
        change_id = int(request.POST.get(action_name))
        login = request.user.username

        try:
            link = self._get_oebs_link_usecase.position_link(obj.id, change_id, login, False)
            return HttpResponseRedirect(link)
        except OEBSError as e:
            self.message_user(request, str(e), messages.ERROR)
            return HttpResponseRedirect(redirect_url)

    def _edit_position(self, request, obj, redirect_url, action_name):
        change_id = int(request.POST.get(action_name))
        login = request.user.username

        try:
            link = self._get_oebs_link_usecase.position_link(obj.id, change_id, login, True)
            return HttpResponseRedirect(link)
        except OEBSError as e:
            self.message_user(request, str(e), messages.ERROR)
            return HttpResponseRedirect(redirect_url)

    def _view_budget(self, request, obj, redirect_url, action_name):
        change_id = int(request.POST.get(action_name))
        try:
            link = self._get_oebs_link_usecase.budget_link(obj.id, change_id)
            return HttpResponseRedirect(link)
        except OEBSError as e:
            self.message_user(request, str(e), messages.ERROR)
            return HttpResponseRedirect(redirect_url)

    def response_post_save_change(self, request, obj):
        actions = {
            '_mark_as_processed_manually': self._mark_as_processed_manually,
            '_send_to_oebs': self._send_to_oebs,
            '_resend_failed_to_oebs': self._resend_failed_to_oebs,
            '_confirm_workflow': self._confirm_workflow,
            '_view_position': self._view_position,
            '_edit_position': self._edit_position,
            '_view_budget': self._view_budget,
            '_cancel': self._cancel,
        }

        custom_action = [x for x in actions if x in request.POST] + [None]
        custom_action = custom_action[0]

        if not custom_action:
            return super().response_post_save_change(request, obj)

        list(messages.get_messages(request))  # clean messages
        redirect_url = add_preserved_filters(
            {'preserved_filters': self.get_preserved_filters(request), 'opts': self.model._meta}, request.path
        )

        return actions[custom_action](request, obj, redirect_url, custom_action)


def allow_tags(method):
    method.allow_tags = True
    return method


def hide_field_without_value(method):
    @wraps(method)
    def wrapper(self, obj: ChangeRegistry) -> Optional[str]:
        value = method(self, obj)
        # тут добавляем стили для скрытия дива с этим полем, если функция вернула None
        # проверка obj.id нужна, т.к. джанга добавляет дополнительный пустой инстанс в inline форму
        if value:
            return value

        first_change = obj.workflow.changeregistry_set.first()
        if first_change:
            number = obj.id - first_change.id
        else:
            number = 0

        magic_style = '<style> #changeregistry_set-%d > fieldset > .field-%s { display: none; } </style>'
        return magic_style % (number, method.__name__)

    wrapper.allow_tags = True
    return wrapper


def short_description(description: str):
    def wrapper(method):
        method.short_description = description
        return method
    return wrapper


class ChangeMetaFields:
    def has_add_permission(self, request):
        return False

    def has_delete_permission(self, request, obj=None):
        return False

    def get_readonly_fields(self, request, obj=None):
        fields = super().get_readonly_fields(request, obj)
        for fieldset in self.fieldsets:
            for field in fieldset[1]['fields']:
                if isinstance(field, (list, tuple)):
                    fields += field
                else:
                    fields += (field,)
        return fields

    def get_queryset(self, request):
        qs = super().get_queryset(request)
        return qs.select_related(
            'staff_bonus_scheme',
            'budget_position',
            'compensation_scheme',
            'department',
            'geo',
            'oebs_grade',
            'office',
            'organization',
            'placement',
            'pay_system',
            'position',
            'review_scheme',
        )

    @hide_field_without_value
    @short_description('Подразделение')
    def get_department(self, obj):
        return obj.department and '<a href="/departments/{url}/">{name}</a>'.format(
            url=obj.department.url, name=obj.department.name,
        )

    @hide_field_without_value
    @short_description('Юр. лицо')
    def get_organization(self, obj):
        return obj.placement and obj.organization and obj.organization.name

    @hide_field_without_value
    @short_description('Расположение')
    def get_placement(self, obj):
        if (obj.organization or obj.office) and not obj.placement:
            return '<span style="color:red">Не удалось определить</span>'
        return obj.placement and obj.placement.name

    @hide_field_without_value
    @short_description('Система оплаты труда')
    def get_pay_system(self, obj):
        return obj.pay_system and obj.pay_system.description

    @allow_tags
    @short_description('Номер БП')
    def get_budget_position(self, obj):
        if obj.budget_position:
            return obj.budget_position.code

        return '<span style="color:red">Не определен</span>'

    @hide_field_without_value
    @short_description('Логин')
    def get_login(self, obj: ChangeRegistry):
        if obj.staff is None:
            return None

        return '<a href="https://{url}/{login}">{login}</a>'.format(
            url=settings.STAFF_HOST,
            login=obj.staff.login,
        )

    @hide_field_without_value
    @short_description('Должность')
    def get_position(self, obj: ChangeRegistry):
        return obj and obj.position.name

    @hide_field_without_value
    @short_description('Номер новой или созданной БП')
    def get_new_budget_position(self, obj):
        return obj.new_budget_position and obj.new_budget_position.code

    @hide_field_without_value
    @short_description('Номер кредитной БП')
    def get_linked_budget_position(self, obj):
        return obj.linked_budget_position and obj.linked_budget_position.code

    @hide_field_without_value
    @short_description('Статус транзакции')
    def get_push_status(self, obj):
        return obj.push_status and PUSH_STATUS[obj.push_status] or '—'

    @hide_field_without_value
    @short_description('Тикет')
    def get_ticket(self, obj):
        if obj.ticket:
            return '<a href="{url}/{ticket}">{name}</a>'.format(
                url=settings.STARTREK_URL, ticket=obj.ticket, name=obj.ticket,
            )

    @hide_field_without_value
    @short_description('Статус')
    def get_position_type(self, obj):
        return obj.position_type and obj.position_type.upper()

    @hide_field_without_value
    @short_description('Оклад')
    def get_salary(self, obj):
        if obj.salary:
            return '{:,.2f}'.format(obj.salary).replace(',', ' ').replace('.', ',')

    @hide_field_without_value
    @short_description('Валюта оклада')
    def get_currency(self, obj):
        return obj.currency

    @hide_field_without_value
    @short_description('Ставка')
    def get_rate(self, obj):
        if obj.rate:
            return '{:.3f}'.format(obj.rate).replace('.', ',')

    @hide_field_without_value
    @short_description('Дата действия')
    def get_effective_date(self, obj):
        return obj.effective_date

    @hide_field_without_value
    @short_description('Дата увольнения')
    def get_dismissal_date(self, obj):
        return obj.dismissal_date

    @hide_field_without_value
    @short_description('Схема премий')
    def get_bonus(self, obj):
        return obj.staff_bonus_scheme and obj.staff_bonus_scheme.name

    @hide_field_without_value
    @short_description('Схема компенсаций')
    def get_compensation(self, obj):
        return obj.compensation_scheme and obj.compensation_scheme.name

    @hide_field_without_value
    @short_description('Схема ревью')
    def get_review(self, obj):
        return obj.review_scheme and obj.review_scheme.name

    @hide_field_without_value
    @short_description('География')
    def get_geography(self, obj):
        return obj.geo and f'{obj.geo.name} {obj.geo.oebs_code}'

    @hide_field_without_value
    @short_description('Грейд')
    def get_level(self, obj):
        if obj.oebs_grade:
            return obj.oebs_grade.level

        return None

    @hide_field_without_value
    @short_description('Профессия')
    def get_profession(self, obj):
        if obj.oebs_grade:
            return obj.oebs_grade.occupation.description

        return None

    @hide_field_without_value
    @short_description('HR Продукт')
    def get_hr_product(self, obj):
        product = obj.staff_hr_product
        if not product:
            return None

        return f'{product.product_name}({product.id})'

    @hide_field_without_value
    @short_description('Статус БП')
    def get_bp_status(self, obj):
        return '<span style="color:red">Будет закрыта</span>' if obj.remove_budget_position else None


class ChangeRegistryInline(ChangeMetaFields, admin.StackedInline):
    model = ChangeRegistry
    ordering = ('id',)
    extra = 0
    verbose_name = ''
    verbose_name_plural = 'Изменения БП'
    can_delete = False

    def has_add_permission(self, request):
        return False

    def has_delete_permission(self, request, obj=None):
        return False

    fieldsets = (
        (
            None,
            {
                'fields': (
                    'get_budget_position',
                    'get_login',
                    'get_new_budget_position',
                    'get_linked_budget_position',
                    'get_actions',
                    'get_push_status',
                    'last_oebs_error',
                ),
            },
        ),
        (
            'Изменяемые данные',
            {
                'fields': (
                    'get_ticket',
                    'get_position_type',
                    'get_department',
                    'get_organization',
                    'get_placement',
                    'get_hr_product',
                    'get_profession',
                    'get_level',
                    'get_pay_system',
                    'get_salary',
                    'get_currency',
                    'get_rate',
                    'get_review',
                    'get_bonus',
                    'get_compensation',
                    'get_geography',
                    'get_dismissal_date',
                    'get_bp_status',
                    'get_effective_date',
                    'get_position',
                ),
            },
        ),
    )

    @hide_field_without_value
    @short_description('Действия с БП')
    def get_actions(self, obj):
        if obj.budget_position is None and obj.new_budget_position is None:
            return None

        button = (
            '<button type="submit" name="{action}" value="{obj_id}" formtarget="_blank" style="margin-right:10px">'
            '{name}'
            '</button>'
        )

        edit_position_button = button.format(name='Редактирование', action='_edit_position', obj_id=obj.id)
        view_position_button = button.format(name='Просмотр', action='_view_position', obj_id=obj.id)

        view_budget_button = button.format(name='Корректировка', action='_view_budget', obj_id=obj.id)
        if not obj.correction_id:
            view_budget_button = ''

        return edit_position_button + view_position_button + view_budget_button


class WorkflowAdmin(TemplateForWorkflowRegistry, admin.ModelAdmin):
    model = Workflow

    list_display = (
        'id',
        'created_at',
        'code',
        'get_status',
        'get_bp_codes',
    )

    fields = (
        'id',
        'created_at',
        'code',
        'catalyst',
        'get_status',
        'get_info',
        'get_bp_codes',
        'get_proposal',
        'get_vacancy',
    )

    readonly_fields = fields

    inlines = (ChangeRegistryInline,)

    list_filter = (
        'code',
        'status',
        'manually_processed',
    )

    search_fields = (
        'id',
        'changeregistry__budget_position__code',
        'changeregistry__ticket',
        'changeregistry__staff__login',
        'proposal__proposal_id',
        'changeregistry__oebs_transaction_id',
    )

    actions = (
        'push_workflows_to_oebs',
        'push_failed_workflows_to_oebs',
        'confirm_credit_management_workflows',
    )

    @describe('Отправить выбранные Workflow в OEBS')
    def push_workflows_to_oebs(self, request, queryset):
        if set(queryset.values_list('status', flat=True)) != {WORKFLOW_STATUS.CONFIRMED}:
            self.message_user(
                request=request,
                message='Действие выполнимо только для Workflow в статусе "Согласовано".',
                level=messages.ERROR,
            )
            return

        workflow_service = WorkflowRegistryService()
        workflow_ids = list(str(wf.id) for wf in queryset)
        workflow_service.repository.queue_workflows(workflow_ids, request.user.get_profile())
        self.message_user(request, 'Создана задача на отправку выбранных Workflow в OEBS.', messages.SUCCESS)

    @describe('Повторно отправить выбранные Workflow в OEBS')
    @atomic()
    def push_failed_workflows_to_oebs(self, request, queryset):
        if set(queryset.values_list('status', flat=True)) != {WORKFLOW_STATUS.FINISHED}:
            self.message_user(
                request=request,
                message='Действие выполнимо только для Workflow в статусе "Отправлено".',
                level=messages.ERROR,
            )
            return

        workflow_service = WorkflowRegistryService()
        workflow_ids = list(str(wf.id) for wf in queryset)
        workflow_service.push_failed_workflows_to_oebs(workflow_ids, request.user.get_profile().id)
        self.message_user(request, 'Создана задача только для неотправленных ранее Workflow в OEBS.', messages.SUCCESS)

    @describe('Массовая отправка заявок на закрытие кредита')
    @atomic()
    def confirm_credit_management_workflows(self, request, queryset):
        if set(queryset.values_list('status', flat=True)) != {WORKFLOW_STATUS.PENDING}:
            self.message_user(
                request=request,
                message='Действие выполнимо только для Credit Management Workflow в статусе "В ожидании".',
                level=messages.ERROR,
            )
            return

        if queryset.exclude(code__in=CAN_CONFIRM_CREDIT_MANAGEMENT_CODES).exists():
            self.message_user(
                request=request,
                message='Действие выполнимо только для Credit Management Workflow.',
                level=messages.ERROR,
            )
            return

        workflow_service = WorkflowRegistryService()
        workflow_ids = list(wf.id for wf in queryset)
        workflow_service.confirm_workflows(workflow_ids)
        self.message_user(request, 'Создана задача на отправку в OEBS.', messages.SUCCESS)

    def has_add_permission(self, request):
        return False

    def has_delete_permission(self, request, obj=None):
        return False

    def get_queryset(self, request):
        queryset = super(WorkflowAdmin, self).get_queryset(request)
        return queryset.prefetch_related('changeregistry_set').select_related('vacancy', 'proposal', 'catalyst')

    def get_ordering(self, request):
        return ['-created_at']

    def get_bp_codes(self, obj):
        field = 'budget_position__code'
        bp_codes = obj.changeregistry_set.distinct(field).values_list(field, flat=True)
        if not bp_codes:
            return '—'
        return ', '.join(map(str, bp_codes))

    get_bp_codes.short_description = 'Изменяемые БП'

    def get_proposal(self, obj):
        return (
            obj.proposal
            and '<a href="/proposal/{id}/">{name}</a>'.format(
                id=obj.proposal.proposal_id, name=obj.proposal.proposal_id,
            )
            or '—'
        )

    get_proposal.short_description = 'Proposal'
    get_proposal.allow_tags = True

    def get_vacancy(self, obj):
        return (
            obj.vacancy
            and '<a href="https://{host}/vacancies/{vacancy}/">{name}</a>'.format(
                host=settings.FEMIDA_HOST, vacancy=obj.vacancy.id, name=obj.vacancy.name,
            )
            or '—'
        )

    get_vacancy.short_description = 'Вакансия'
    get_vacancy.allow_tags = True

    def get_status(self, obj):
        if not obj.status:
            return '—'

        human_status = WORKFLOW_STATUS[obj.status]

        if obj.status != WORKFLOW_STATUS.FINISHED:
            return human_status

        if obj.manually_processed:
            return '{} вручную'.format(human_status)

        if obj.changeregistry_set.filter(push_status=PUSH_STATUS.ERROR).exists():
            return '{} с ошибкой'.format(human_status)

        return '{} успешно'.format(human_status)

    get_status.short_description = 'Статус'

    def get_info(self, obj):
        completed_statuses = [WORKFLOW_STATUS.PUSHED, WORKFLOW_STATUS.FINISHED, WORKFLOW_STATUS.SENDING_NOTIFICATION]
        if obj.status in completed_statuses and obj.catalyst:
            return f'Отправлено от лица <a href="/{obj.catalyst.login}/">{obj.catalyst.login}</a>'
        return '—'

    get_info.short_description = 'Доп. данные'
    get_info.allow_tags = True


class GradeAdmin(admin.ModelAdmin):
    list_display = (
        'grade_id',
        'level',
        'occupation',
    )

    search_fields = (
        'occupation',
    )


class ProfLevelMappingForm(forms.ModelForm):
    mapping_fields = (
        'intern',
        'junior',
        'middle',
        'senior',
        'lead',
        'expert',
    )

    class Meta:
        model = ProfLevelMapping
        fields = (
            'occupation',
            'intern',
            'junior',
            'middle',
            'senior',
            'lead',
            'expert',
        )

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.fields['occupation'].queryset = Occupation.objects.order_by('pk')

    def clean(self):
        cleaned_data = super().clean()
        occupation = cleaned_data['occupation']
        for field in self.mapping_fields:
            value = cleaned_data.get(field)
            if value is not None:
                if not OEBSGrade.objects.filter(occupation=occupation, level=value).exists():
                    raise ValidationError(f'Уровень грейда {value} для профессии {occupation.pk} не существует')

        return cleaned_data


class ProfLevelMappingAdmin(admin.ModelAdmin):
    list_display = (
        'occupation',
        'intern',
        'junior',
        'middle',
        'senior',
        'lead',
        'expert',
    )

    search_fields = (
        'occupation',
    )

    form = ProfLevelMappingForm


class MarketWorkflowAdmin(WorkflowAdmin):
    class MarketWorkflow(Workflow):
        class Meta:
            proxy = True

    def get_queryset(self, request):
        queryset = super().get_queryset(request)

        return queryset.filter(changeregistry__in=self._get_market_changes(request)).distinct()

    @describe('Отправить выбранные Workflow в OEBS')
    def push_workflows_to_oebs(self, request, queryset) -> None:
        self._wrap_market_call(
            request,
            queryset,
            super().push_workflows_to_oebs,
        )

    @describe('Повторно отправить выбранные Workflow в OEBS')
    def push_failed_workflows_to_oebs(self, request, queryset) -> None:
        self._wrap_market_call(
            request,
            queryset,
            super().push_failed_workflows_to_oebs,
        )

    @describe('Массовая отправка заявок на закрытие кредита')
    def confirm_credit_management_workflows(self, request, queryset) -> None:
        self._wrap_market_call(
            request,
            queryset,
            super().confirm_credit_management_workflows,
        )

    def _send_to_oebs(self, request, obj, redirect_url, _):
        return self._wrap_market_call_on_object(
            request,
            obj,
            redirect_url,
            super()._send_to_oebs,
        )

    def _resend_failed_to_oebs(self, request, obj, redirect_url, _):
        return self._wrap_market_call_on_object(
            request,
            obj,
            redirect_url,
            super()._resend_failed_to_oebs,
        )

    def _mark_as_processed_manually(self, request, obj, redirect_url, _):
        return self._wrap_market_call_on_object(
            request,
            obj,
            redirect_url,
            super()._mark_as_processed_manually,
        )

    def _confirm_workflow(self, request, obj, redirect_url, _):
        return self._wrap_market_call_on_object(
            request,
            obj,
            redirect_url,
            super()._confirm_workflow,
        )

    def _cancel(self, request, obj, redirect_url, _):
        return self._wrap_market_call_on_object(
            request,
            obj,
            redirect_url,
            super()._cancel,
        )

    def _get_market_changes(self, request):
        return ChangeRegistry.objects.filter(self._get_market_departments_filter(request))

    @staticmethod
    def _get_market_departments_filter(request):
        department_filter = request.user.get_profile().departments_by_perm_query(
            perm='budget_position.can_manage_market_budget_positions',
            by_children=True,
            instance_classes={InstanceClass.DEPARTMENT.value},
        )

        market_departments = Department.objects.filter(department_filter)

        return Q(department__in=market_departments) | Q(staff__department__in=market_departments)

    def _wrap_market_call(self, request, queryset, call):
        market_workflow_count = queryset.filter(changeregistry__in=self._get_market_changes(request)).distinct().count()

        if market_workflow_count != queryset.distinct().count():
            self._show_generic_error(request)
            return

        call(request, queryset)

    def _wrap_market_call_on_object(self, request, obj, redirect_url, call):
        if not obj.changeregistry_set.filter(self._get_market_departments_filter(request)).exists():
            self._show_generic_error(request)
            return HttpResponseRedirect(redirect_url)

        return call(request, obj, redirect_url, None)

    def _show_generic_error(self, request):
        self.message_user(
            request=request,
            message='Действие доступно только для Workflow по подразделениям и сотрудникам "Яндекс.Маркет".',
            level=messages.ERROR,
        )


class GeoMappingAdmin(admin.ModelAdmin):
    list_display = ('staff_geography', 'oebs_geography_code')


class PersonSchemeExceptionsAdmin(BaseIntranetAdmin, AutocompleteMixin):
    list_display = ('person', 'reward_scheme_id', 'bonus_scheme_id', 'review_scheme_id', 'intranet_status')
    staff_field = 'person'

    def formfield_for_foreignkey(self, db_field, request=None, **kwargs):
        autocomplete_field = self.maybe_autocomplete_formfield(db_field)
        if autocomplete_field is not None:
            return autocomplete_field

        return super().formfield_for_foreignkey(db_field, request, **kwargs)


class RewardModelAdmin(admin.ModelAdmin):
    list_display = ('scheme_id', 'name', 'description', 'start_date', 'end_date')
    search_fields = ('scheme_id', 'name', 'description')
    ordering = ('scheme_id', 'start_date')


class ReviewModelAdmin(admin.ModelAdmin):
    list_display = ('scheme_id', 'name', 'description', 'start_date', 'end_date')
    search_fields = ('scheme_id', 'name', 'description')
    ordering = ('scheme_id', 'start_date')


admin.site.register(GeographyMapping, GeoMappingAdmin)
admin.site.register(MarketWorkflowAdmin.MarketWorkflow, MarketWorkflowAdmin)
admin.site.register(OEBSGrade, GradeAdmin)
admin.site.register(PersonSchemeExceptions, PersonSchemeExceptionsAdmin)
admin.site.register(ProfLevelMapping, ProfLevelMappingAdmin)
admin.site.register(Review, ReviewModelAdmin)
admin.site.register(Reward, RewardModelAdmin)
admin.site.register(Workflow, WorkflowAdmin)
