import attr


@attr.s
class Result(object):
    has_result = attr.ib(type=bool)
    result = attr.ib()

    @classmethod
    def empty(cls):
        return Result(False, None)


class Node(object):
    def _calc_result(self):
        raise NotImplementedError

    def collect(self):
        calc_result = self._calc_result()
        return calc_result.result if calc_result.has_result else None


class AndValueIsTrue(Node):
    def __init__(self, entry_node):
        self.entry_node = entry_node

    def _calc_result(self):
        calc_result = self.entry_node._calc_result()
        if not calc_result.has_result:
            return Result.empty()

        if calc_result.result:
            return calc_result

        return Result.empty()

    def then(self, f):
        return ThenNode(self, f)

    def then_collect(self, f):
        return self.then(f).collect()

    def or_value(self, value):
        return OrValueNode(self, value)


class ThenNode(Node):
    def __init__(self, entry_node, f):
        self.entry_node = entry_node
        self.f = f

    def _calc_result(self):
        calc_result = self.entry_node._calc_result()
        if calc_result.has_result:
            return Result(True, self.f(calc_result.result))
        return Result.empty()

    def or_value(self, value):
        return OrValueNode(self, value)


class OrValueNode(Node):
    def __init__(self, then_node, value):
        self.then_node = then_node
        self.value = value

    def _calc_result(self):
        calc_result = self.then_node._calc_result()
        if calc_result.has_result:
            return calc_result
        return Result(True, self.value)


class EntryNode(Node):
    def __init__(self, d, key):
        self.d = d
        self.key = key

    def then(self, f):
        return ThenNode(self, f)

    def then_collect(self, f):
        return self.then(f).collect()

    def and_value(self):
        return AndValueIsTrue(self)

    def _calc_result(self):
        return Result(True, self.d[self.key]) if self.key in self.d else Result.empty()


def entry(d, key):
    return EntryNode(d, key)


def entry_api(d, value_should_be_true=True):
    def wrap(key):
        if value_should_be_true:
            return entry(d, key).and_value()

        return entry(d, key)
    return wrap
