from typing import Set

import sform

from django.core.exceptions import ValidationError

from staff.departments.models import Department
from staff.person.models import Occupation


class TableflowBaseValidationForm(sform.SForm):
    def human_readable_errors(self):
        CASES_FIELD = 'cases'
        result = []
        for error_coordinates, validation_errors in self._errors.items():
            case_idx = None
            field = None

            if len(error_coordinates) >= 3 and error_coordinates[0] == CASES_FIELD:
                case_idx = error_coordinates[1]
                field = error_coordinates[3]

            for validation_error in validation_errors:
                result.append(f'{validation_error.message} at row {case_idx} at field {field}')

        return result


class RulesWithDepartmentValidationForm(TableflowBaseValidationForm):
    def clean(self):
        cleaned_data = super().clean()
        if 'cases' not in cleaned_data:
            raise ValidationError('Should be at least one rule')

        any_department_rule = 'any'
        for case in cleaned_data['cases']:
            department_url_rule = case['checks']['department_url']
            if department_url_rule['ne'].lower() == any_department_rule:
                return cleaned_data

        raise ValidationError('Should be at least one rule with department !=Any to cover all possible variants')


class MultipleOperatorsValidationForm(sform.SForm):
    def is_not_empty_value(self, val) -> bool:
        if isinstance(val, int):
            return True

        if isinstance(val, str):
            return bool(val)

        if isinstance(val, list):
            return bool(val) and all(self.is_not_empty_value(list_val) for list_val in val)

        return False

    def clean(self):
        cleaned_data = super().clean()
        operator_args = [cleaned_data.get(field_name) for field_name in self.fields.keys()]
        provided_values_len = len([
            arg
            for arg in operator_args
            if self.is_not_empty_value(arg)
        ])
        provided_fields = {
            field_name
            for field_name in self.fields.keys()
            if self.is_not_empty_value(cleaned_data.get(field_name))
        }
        provided_more_than_one_operator = provided_values_len > 1

        if provided_more_than_one_operator:
            self.on_provided_more_than_one_operator(provided_fields)

        nothing_provided = provided_values_len == 0
        if nothing_provided:
            raise ValidationError('Required value')

        return cleaned_data

    def on_provided_more_than_one_operator(self, provided_fields: Set[str]):
        raise NotImplementedError


class EntityValidationForm(MultipleOperatorsValidationForm):
    eq = sform.CharField(state=sform.NORMAL)
    ne = sform.CharField(state=sform.NORMAL)
    contains = sform.GridField(sform.CharField(), state=sform.NORMAL)

    def on_provided_more_than_one_operator(self, provided_fields: Set[str]):
        raise ValidationError('Only one operator should be provided')


class IntergerValidationForm(MultipleOperatorsValidationForm):
    eq = sform.CharField(state=sform.NORMAL)
    ne = sform.CharField(state=sform.NORMAL)
    contains = sform.GridField(sform.CharField(), state=sform.NORMAL)
    le = sform.CharField(state=sform.NORMAL)
    ge = sform.CharField(state=sform.NORMAL)
    lt = sform.CharField(state=sform.NORMAL)
    gt = sform.CharField(state=sform.NORMAL)

    def on_provided_more_than_one_operator(self, provided_fields: Set[str]):
        between_operators_set = {'ge', 'le'}
        if provided_fields == between_operators_set:
            return

        raise ValidationError('Only one operator should be provided')


def integer_field():
    return sform.FieldsetField(IntergerValidationForm, state=sform.REQUIRED)


class DepartmentValidationForm(EntityValidationForm):
    @staticmethod
    def clean_eq(department_url):
        if not department_url:
            return department_url

        if not Department.objects.filter(intranet_status=1, url=department_url).exists():
            raise ValidationError(f'Invalid department url {department_url}')
        return department_url

    @staticmethod
    def clean_contains(department_urls):
        if not department_urls:
            return department_urls

        department_urls_set = set(department_urls)
        existing_department_urls_set = set(
            Department.objects
            .filter(intranet_status=1, url__in=department_urls)
            .values_list('url', flat=True)
        )
        if len(department_urls_set) > len(existing_department_urls_set):
            wrong_urls = department_urls_set - existing_department_urls_set
            raise ValidationError(f'invalid department urls {wrong_urls}')

        return department_urls


def department_field():
    return sform.FieldsetField(DepartmentValidationForm, state=sform.REQUIRED)


class OccupatonValidationForm(EntityValidationForm):
    @staticmethod
    def clean_eq(occupation_code):
        if not occupation_code:
            return occupation_code

        if not Occupation.objects.filter(pk=occupation_code).exists():
            raise ValidationError(f'Invalid occupation {occupation_code}')
        return occupation_code

    @staticmethod
    def clean_contains(occupation_codes):
        if not occupation_codes:
            return occupation_codes

        occupaton_codes_set = set(occupation_codes)
        existing_occupation_codes_set = set(
            Occupation.objects
            .filter(pk__in=occupaton_codes_set)
            .values_list('pk', flat=True)
        )
        if len(occupaton_codes_set) > len(existing_occupation_codes_set):
            wrong_codes = occupaton_codes_set - existing_occupation_codes_set
            raise ValidationError(f'Invalid occupation codes {wrong_codes}')

        return occupation_codes


def occupation_field():
    return sform.FieldsetField(OccupatonValidationForm, state=sform.REQUIRED)


class BonusGroupRulesOutForm(sform.SForm):
    bonus_id = sform.CharField(state=sform.REQUIRED)
    priority = sform.CharField(state=sform.REQUIRED)


class BonusGroupChecksForm(sform.SForm):
    bonus_group = sform.FieldsetField(EntityValidationForm, state=sform.REQUIRED)
    department_url = department_field()


class BonusGroupRulesCaseForm(sform.SForm):
    out = sform.FieldsetField(BonusGroupRulesOutForm, state=sform.REQUIRED)
    checks = sform.FieldsetField(BonusGroupChecksForm, state=sform.REQUIRED)


class BonusGroupRulesValidationForm(RulesWithDepartmentValidationForm):
    cases = sform.GridField(sform.FieldsetField(BonusGroupRulesCaseForm), state=sform.REQUIRED)


class RewardCategoryRulesOutForm(sform.SForm):
    category = sform.CharField(state=sform.REQUIRED)


class RewardCategoryCheckForm(sform.SForm):
    occupation_reward_group = sform.FieldsetField(EntityValidationForm, state=sform.REQUIRED)


class RewardCategoryRulesCaseForm(sform.SForm):
    out = sform.FieldsetField(RewardCategoryRulesOutForm, state=sform.REQUIRED)
    checks = sform.FieldsetField(RewardCategoryCheckForm, state=sform.REQUIRED)


class RewardCategoryRulesValidationForm(TableflowBaseValidationForm):
    cases = sform.GridField(sform.FieldsetField(RewardCategoryRulesCaseForm))


class ReviewGroupCheckForm(sform.SForm):
    review_group = sform.FieldsetField(EntityValidationForm, state=sform.REQUIRED)
    department_url = department_field()
    grade_level = integer_field()


class ReviewGroupRulesOutForm(sform.SForm):
    review_id = sform.CharField(state=sform.REQUIRED)
    priority = sform.CharField(state=sform.REQUIRED)


class ReviewGroupRulesCaseForm(sform.SForm):
    out = sform.FieldsetField(ReviewGroupRulesOutForm, state=sform.REQUIRED)
    checks = sform.FieldsetField(ReviewGroupCheckForm, state=sform.REQUIRED)


class ReviewGroupRulesValidationForm(RulesWithDepartmentValidationForm):
    cases = sform.GridField(sform.FieldsetField(ReviewGroupRulesCaseForm))


class RewardCheckForm(sform.SForm):
    category = sform.FieldsetField(EntityValidationForm, state=sform.REQUIRED)
    department_url = department_field()
    is_internship = integer_field()


class RewardRulesOutForm(sform.SForm):
    reward_scheme_id = sform.CharField(state=sform.REQUIRED)
    priority = sform.CharField(state=sform.REQUIRED)


class RewardRulesCaseForm(sform.SForm):
    out = sform.FieldsetField(RewardRulesOutForm, state=sform.REQUIRED)
    checks = sform.FieldsetField(RewardCheckForm, state=sform.REQUIRED)


class RewardRulesValidationForm(RulesWithDepartmentValidationForm):
    cases = sform.GridField(sform.FieldsetField(RewardRulesCaseForm))


class BonusCheckForm(sform.SForm):
    grade_level = integer_field()
    occupation_code = occupation_field()
    department_url = department_field()


class BonusRulesOutForm(sform.SForm):
    bonus_id = sform.CharField(state=sform.REQUIRED)
    priority = sform.CharField(state=sform.REQUIRED)


class BonusRulesCaseForm(sform.SForm):
    out = sform.FieldsetField(BonusRulesOutForm, state=sform.REQUIRED)
    checks = sform.FieldsetField(BonusCheckForm, state=sform.REQUIRED)


class BonusRulesValidationForm(TableflowBaseValidationForm):
    cases = sform.GridField(sform.FieldsetField(BonusRulesCaseForm))


class ReviewRulesOutForm(sform.SForm):
    review_id = sform.CharField(state=sform.REQUIRED)
    priority = sform.CharField(state=sform.REQUIRED)


class ReviewCheckForm(sform.SForm):
    occupation_code = occupation_field()
    department_url = department_field()
    grade_level = integer_field()


class ReviewRulesCaseForm(sform.SForm):
    out = sform.FieldsetField(ReviewRulesOutForm, state=sform.REQUIRED)
    checks = sform.FieldsetField(ReviewCheckForm, state=sform.REQUIRED)


class ReviewRulesValidationForm(TableflowBaseValidationForm):
    cases = sform.GridField(sform.FieldsetField(ReviewRulesCaseForm))
