import logging
from uuid import UUID
from ids.exceptions import EmptyIteratorError

from django.conf import settings
from staff.celery_app import app
from staff.lib.log import log_context
from staff.lib.tasks import LOCK_TIME, LockedTask
from staff.lib.startrek import issues
from staff.person.models import Staff

from staff.budget_position.const import WORKFLOW_STATUS
from staff.budget_position.models import Workflow
from staff.budget_position.workflow_service.gateways import (
    CreditManagementService,
    FemidaService,
    WorkflowRepository,
    StartrekService,
    StaffService,
)
from staff.budget_position.workflow_service import workflow_registry_service


logger = logging.getLogger(__name__)


@app.task(ignore_result=True)
class UpdateWorkflowPushStatus(LockedTask):
    def locked_run(self, workflow_id):
        from staff.budget_position.workflow_service.workflow_registry_service import WorkflowRegistryService
        WorkflowRegistryService().try_update_workflow_push_status(workflow_id)

    def get_lock_name(self, workflow_id, *args):
        return 'update_workflow_push_status_{}'.format(workflow_id)


def update_push_status(workflow_id):
    # По задумке эта функция вызывается из той же таски, которая берет лок,
    # а все таски с локом работают минимум LOCK_TIME секунд:
    # https://github.yandex-team.ru/tools-sox/staff/blob/1.14.208/staff/lib/tasks.py#L51
    UpdateWorkflowPushStatus.apply_async(countdown=LOCK_TIME + 5, args=[str(workflow_id)])


@app.task(ignore_result=True)
class RunWorkflowsStatusUpdating(LockedTask):
    def locked_run(self):
        from django.db.models import Q
        suspect_filter = Q(status__in=(WORKFLOW_STATUS.PUSHED, WORKFLOW_STATUS.SENDING_NOTIFICATION))
        for workflow_id in Workflow.objects.filter(suspect_filter).values_list('id', flat=True):
            UpdateWorkflowPushStatus.delay(str(workflow_id))


@app.task(ignore_result=True)
class PushToFemida(LockedTask):
    def locked_run(self, *args, **kwargs):
        FemidaService(WorkflowRepository()).push_scheduled()


@app.task(ignore_result=True)
class PushWorkflow(LockedTask):
    def locked_run(self):
        from staff.budget_position.workflow_service.use_cases import PushWorkflow

        repo = WorkflowRepository()
        use_case = PushWorkflow(
            repository=repo,
            startrek_service=StartrekService(),
            staff_service=StaffService(),
            femida_service=FemidaService(repo),
            credit_management_service=CreditManagementService(repo),
        )

        for workflow_id in use_case.get_workflows(WORKFLOW_STATUS.QUEUED):
            with log_context(workflow_id=str(workflow_id)):
                try:
                    use_case.push_workflow(workflow_id)
                except Exception as e:
                    logger.exception(f'Unhandled exception on pushing workflow {workflow_id}: {e}')

        for workflow_id in use_case.get_workflows(WORKFLOW_STATUS.SENDING_NOTIFICATION):
            with log_context(workflow_id=str(workflow_id)):
                try:
                    use_case.try_send_notification_for_workflow(workflow_id)
                except Exception as e:
                    logger.exception(f'Unhandled exception on notifying workflow {workflow_id}: {e}')


@app.task(ignore_result=True)
class PushFromStartrekToOEBS(LockedTask):
    def locked_run(self, ticket):
        try:
            issue = issues.get_issue(ticket)
        except EmptyIteratorError:
            logger.error('Startrek can\'t find ticket %s', ticket)
            return

        analyst = Staff.objects.filter(login=issue.analyst.login).first()
        workflow_id_str = str(issue[settings.STARTREK_WORKFLOW_FIELD_ID])
        if not analyst or not workflow_id_str:
            return
        workflow_id = UUID(workflow_id_str)

        workflow_registry_service.WorkflowRegistryService().push_workflow_to_oebs(workflow_id, analyst.id)
