from typing import Dict, Tuple


from staff.budget_position.workflow_service import errors, UnexpectedResponseOEBSError


class FemidaErrors:
    def make_form_error_response(self, sform_errors: Dict) -> Dict:
        """
        На вход принимает словарь ошибок из SForm
        Отдает ошибку на русском и английском
        Меняет словарь с ошибками
        """
        errors_sections = sform_errors['errors']
        messages_ru = []
        messages_en = []

        if self._has_budget_position_id_error(errors_sections):
            message_ru, message_en = self._budget_position_id_error_messages()
            messages_ru.append(message_ru)
            messages_en.append(message_en)

        if self._has_hr_product_translation_id_error(errors_sections):
            message_ru, message_en = self._hr_product_translation_id_error_messages()
            messages_ru.append(message_ru)
            messages_en.append(message_en)

        errors_sections['messages_ru'] = messages_ru
        errors_sections['messages_en'] = messages_en
        return sform_errors

    def make_workflow_resolve_error_response(self, e: errors.AbstractWorkflowResolveError) -> Dict:
        result = {'errors': {'messages_ru': [], 'messages_en': []}}

        if isinstance(e, errors.EmptyBPOnNonVacancyCreation):
            self._add_error_messages(result, self._empty_bp_on_non_vacancy_creation_error_messages())
        elif isinstance(e, errors.NonOfferRejectionForOfferBp):
            self._add_error_messages(result, self._non_offer_rejection_for_offer_bp())
        elif isinstance(e, errors.NonVacancyCreationForPlanBP):
            self._add_error_messages(result, self._non_vacancy_creation_for_vacancy_plan_bp())
        elif isinstance(e, errors.NonVacancyCreationForOccupiedBP):
            self._add_error_messages(result, self._non_vacancy_creation_for_occupied_bp())
        elif isinstance(e, errors.UnknownBP):
            self._add_error_messages(result, self._unknown_bp())
        elif isinstance(e, errors.NotSupportedBPState):
            self._add_error_messages(result, self._not_supported_bp_state())
        elif isinstance(e, errors.NoHRAnalyst):
            self._add_error_messages(result, self._no_hr_analyst())

        return result

    @staticmethod
    def make_oebs_error(e: UnexpectedResponseOEBSError) -> Dict:
        return {
            'errors': {
                'messages_ru': [f'Ошибка при отправке в OEBS: {e}'],
                'messages_en': [f'Error sending to OEBS: {e}'],
            },
        }

    def _add_error_messages(self, sform_errors: Dict, error_messages: Tuple[str, str]):
        message_ru, message_en = error_messages
        sform_errors['errors']['messages_ru'].append(message_ru)
        sform_errors['errors']['messages_en'].append(message_en)

    def _empty_bp_on_non_vacancy_creation_error_messages(self) -> Tuple[str, str]:
        return (
            'Не был получен код БП. За исключением создания вакансии в кредит БП обязательна',
            'BP code not provided. BP code is required, if not credit vacancy creation',
        )

    def _non_offer_rejection_for_offer_bp(self) -> Tuple[str, str]:
        return (
            'По данным Стаффа БП находится в состоянии OFFER, но получена не отмена оффера',
            'By Staff data BP in OFFER state, offer rejection expected',
        )

    def _non_vacancy_creation_for_vacancy_plan_bp(self) -> Tuple[str, str]:
        return (
            'По данным Стаффа БП в состоянии VACANCY_PLAN, но нет создания вакансии',
            'By staff data BP in VACANCY_PLAN state, but got non vacancy creation request',
        )

    def _non_vacancy_creation_for_occupied_bp(self) -> Tuple[str, str]:
        return (
            'По данным Стаффа БП в состоянии OCCUPIED, но нет создания вакансии',
            'By staff data BP in OCCUPIED state, but got non vacancy creation request',
        )

    def _not_supported_bp_state(self) -> Tuple[str, str]:
        return (
            'БП в состоянии для которого нет известных действий',
            'BP state does not support any action',
        )

    def _no_hr_analyst(self) -> Tuple[str, str]:
        return (
            'Не найден HR аналитик ответственный за департамент',
            'No HR analyst found for the department',
        )

    def _has_budget_position_id_error(self, sform_errors: Dict) -> bool:
        return 'budget_position_id' in sform_errors

    def _budget_position_id_error_messages(self) -> Tuple[str, str]:
        return (
            (
                'Стафф не смог найти бюджетную позицию с заданным кодом. '
                'Код БП неверен, либо БП еще не синхронизирована с OEBS'
            ),
            (
                'Staff can\'t find budget position by code. '
                'BP code is wrong or BP not synchronized with OEBS'
            ),
        )

    def _has_hr_product_translation_id_error(self, sform_errors: Dict) -> bool:
        return 'hr_product_translation_id' in sform_errors

    def _hr_product_translation_id_error_messages(self) -> Tuple[str, str]:
        return (
            (
                'Указание основного продукта (VS) обязательно, если бюджетная позиция не указана'
            ),
            (
                'Value stream is mandatory when budget position is not defined'
            ),
        )

    def _unknown_bp(self) -> Tuple[str, str]:
        return (
            (
                'Стафф не смогу найти бюджетную позицию с заданным кодом в данных предоставленных с OEBS. '
                'Код БП неверен, либо БП еще не синхронизирована с OEBS'
            ),
            (

                'Staff can\'t find budget position by code in OEBS data. '
                'BP code is wrong or BP not synchronized with OEBS'
            ),
        )
