import json
import logging
from typing import Dict, List

from django.http import HttpRequest, HttpResponse, JsonResponse
from django.views.decorators.csrf import csrf_exempt
from django.views.decorators.http import require_http_methods

from staff.lib.decorators import require_permission
from staff.lib.forms import errors as form_errors
from staff.lib.utils.qs_values import extract_related, localize

from staff.budget_position.const import WORKFLOW_STATUS
from staff.budget_position.controllers import attach_comment_to_budget_position_controller
from staff.budget_position.forms import GetBlockedBudgetPositionsForm, BudgetPositionCommentForm
from staff.budget_position.models import Workflow


logger = logging.getLogger(__name__)


def _blocked_bp_info_by_position_codes(codes: List[int]) -> Dict[int, Dict[str, int]]:
    qs = (
        Workflow.objects
        .filter(status=WORKFLOW_STATUS.PENDING, changeregistry__budget_position__code__in=codes)
        .values(
            'proposal__proposal_id',
            'proposal__author__login',
            'proposal__author__first_name',
            'proposal__author__first_name_en',
            'proposal__author__last_name',
            'proposal__author__last_name_en',
            'changeregistry__budget_position__code',
            'changeregistry__ticket',
        )
    )

    result = {}

    for workflow_data in qs:
        bp_data = result.setdefault(
            workflow_data['changeregistry__budget_position__code'],
            {
                'proposal_id': workflow_data['proposal__proposal_id'],
                'author': localize(extract_related(workflow_data, 'proposal__author', pop=False)),
                'tickets': [],
            },
        )

        ticket = workflow_data['changeregistry__ticket']
        if ticket and ticket not in bp_data['tickets']:
            bp_data['tickets'].append(ticket)

    return result


@csrf_exempt
@require_http_methods(['POST'])
def blocked_bp_info(request: HttpRequest) -> HttpResponse:
    try:
        data = json.loads(request.body)
    except ValueError:
        return JsonResponse(data=form_errors.invalid_json_error(request.body), status=400)

    form = GetBlockedBudgetPositionsForm(data)
    if not form.is_valid():
        return JsonResponse(data=form.errors, status=400)

    codes = form.cleaned_data['codes']
    result = _blocked_bp_info_by_position_codes(codes)
    return JsonResponse(data=result)


@require_http_methods(['POST'])
@require_permission('departments.can_view_headcounts')
def attach_comment_view(request) -> HttpResponse:
    try:
        form = BudgetPositionCommentForm(json.loads(request.body))
    except ValueError:
        logger.info('Invalid JSON received', exc_info=True)
        return JsonResponse(data={}, status=400)

    if not form.is_valid():
        return JsonResponse(data=form.errors_as_dict(), status=400)

    attach_comment_to_budget_position_controller(
        request.user.get_profile(),
        form.cleaned_data['budget_position'],
        form.cleaned_data.get('comment'),
    )

    return JsonResponse(data={}, status=204)
