import logging
from typing import List
from decimal import Decimal

import attr

from django.http import HttpResponseForbidden, JsonResponse, HttpResponse
from django.views.decorators.csrf import csrf_exempt
from django.views.decorators.http import require_POST

from staff.lib.decorators import consuming_json
from staff.lib.oebs import wrap_oebs_errors
from staff.person.models import Occupation

from staff.budget_position.const import InitStatus, FemidaRequestType
from staff.budget_position.forms import AssignmentCreateForm, AttachToVacancyForm
from staff.budget_position.entry import entry_api
from staff.budget_position.views.femida_errors import FemidaErrors
from staff.budget_position.views.responses import AttachToVacancyResponse
from staff.budget_position.workflow_service import (
    FemidaData,
    WorkflowRegistryService,
    OebsHireError,
    UnexpectedResponseOEBSError,
    errors,
)

logger = logging.getLogger(__name__)


@require_POST
@consuming_json
@csrf_exempt
@wrap_oebs_errors
def attach_to_vacancy(request, json_data) -> HttpResponse:
    if request.client_application.name != 'femida' and request.user.staff.login != 'robot-femida':
        logger.info(
            'User %s with app %s tries to access attach_to_vacancy',
            request.user.staff.login, request.client_application.name,
        )
        return HttpResponseForbidden()

    form = AttachToVacancyForm(data=json_data)
    if not form.is_valid():
        errs = FemidaErrors().make_form_error_response(form.errors_as_dict())
        vacancy = json_data.get('vacancy')
        ticket = json_data.get('job_issue_key')
        logger.warning(
            'Femida\'s data for workflow is not valid: %s for vacancy %s and ticket %s',
            errs,
            vacancy,
            ticket,
        )
        return JsonResponse(data=errs, status=400)

    service = WorkflowRegistryService()

    cleaned_data = form.cleaned_data
    entry = entry_api(cleaned_data)

    request_type = FemidaRequestType(cleaned_data['request_type'])

    occupation_id = entry('profession_key').then_collect(lambda v: v.name)
    occupation = Occupation.objects.filter(pk=occupation_id).first()

    is_offer = request_type == FemidaRequestType.OFFER
    is_internal_offer = request_type == FemidaRequestType.INTERNAL_OFFER

    hr_product_id = entry('hr_product_translation_id').then_collect(lambda v: v.id)
    if not hr_product_id:
        hr_product_id = entry('hr_product_id').then_collect(lambda v: v.id)

    femida_data = FemidaData(
        bonus_scheme_id=entry('bonus_scheme').then_collect(lambda v: v.scheme_id),
        budget_position_code=entry('budget_position_id').then_collect(lambda v: v.code),
        reward_scheme_id=entry('reward_scheme').then_collect(lambda v: v.scheme_id),
        currency=cleaned_data['currency'],
        department_id=entry('department').then_collect(lambda v: v.id),
        dismissal_date=cleaned_data['dismissal_date'] if request_type == FemidaRequestType.VACANCY else None,
        geography_url=entry('geography_translation_id').then_collect(lambda v: v.department_instance.url),
        occupation_id=occupation and occupation.pk,
        grade_level=cleaned_data['grade'],
        professional_level=cleaned_data['professional_level'],
        hr_product_id=hr_product_id,
        person_id=entry('username').then_collect(lambda v: v.id),
        is_internal_offer=is_internal_offer,
        is_offer_rejection=request_type == FemidaRequestType.OFFER_REJECTION,
        is_offer=is_offer,
        is_vacancy_cancellation=request_type == FemidaRequestType.VACANCY_CANCELLATION,
        is_vacancy=request_type == FemidaRequestType.VACANCY,
        office_id=entry('office').then_collect(lambda v: v.id),
        organization_id=entry('organization').then_collect(lambda v: v.id),
        payment_system=cleaned_data['payment_type'],
        rate=entry('work_hours_weekly').then_collect(lambda v: Decimal(v) / Decimal(40)),
        review_scheme_id=entry('review_scheme').then_collect(lambda v: v.scheme_id),
        salary=cleaned_data['salary'],
        ticket=cleaned_data['job_issue_key'],
        salary_ticket=cleaned_data['salary_issue_key'],
        vacancy_id=cleaned_data['vacancy'],
        vacancy_name=cleaned_data['vacancy_name'],
        is_internship=bool(cleaned_data['is_internship']),
    )

    try:
        workflow_id = service.try_create_workflow_from_femida(femida_data)
    except UnexpectedResponseOEBSError as e:
        return JsonResponse(data=FemidaErrors().make_oebs_error(e), status=400)
    except errors.AbstractWorkflowResolveError as e:
        error_details = FemidaErrors().make_workflow_resolve_error_response(e)
        return JsonResponse(data=error_details, status=400)

    response = AttachToVacancyResponse(
        id=str(workflow_id),
        status=InitStatus.CREATED.value,
    )
    return JsonResponse(data=attr.asdict(response), status=201)


@attr.s(auto_attribs=True)
class AssignmentCreateResponse:
    id: str = attr.ib(converter=str)


@attr.s(auto_attribs=True)
class AssignmentCreationFailedResponse:
    oebs_internal_errors: List[str] = attr.ib(factory=list)


@require_POST
@consuming_json
@csrf_exempt
def create_assignment(request, json_data):
    # This API works as a proxy between Femida and OEBS Hire.
    # We rely on correctness of Femida data and do almost no validation.
    # In return we proxy OEBS Hire errors back to Femida.
    if request.client_application.name != 'femida' and request.user.staff.login != 'robot-femida':
        logger.info(
            'User %s with app %s tries to access create_assignment',
            request.user.staff.login, request.client_application.name,
        )
        return HttpResponseForbidden()

    form = AssignmentCreateForm(data=json_data)
    if not form.is_valid():
        errs = form.errors_as_dict()
        logger.warning(
            'Femida\'s data for workflow is not valid: %s for bp %s and person %s',
            errs,
            json_data.get('budget_position_id'),
            json_data.get('person'),
        )
        return JsonResponse(errs, status=400)

    cleaned_data = form.cleaned_data

    def as_is(*args):
        return {f'{field_name}': cleaned_data[field_name] for field_name in args}

    def resolve_id(*args):
        return {
            f'{field_name}_id': cleaned_data[field_name] and cleaned_data[field_name].id
            for field_name in args
        }

    femida_data = FemidaData(
        is_assignment_creation=True,
        budget_position_code=cleaned_data['budget_position_id'] and cleaned_data['budget_position_id'].code,
        wage_system=cleaned_data['payment_type'],
        instead_of_login=cleaned_data['instead_of'] and cleaned_data['instead_of'].login,
        probation_period_code=cleaned_data['probation_period'],
        position_id=cleaned_data['position'] and cleaned_data['position'].code,
        physical_entity_id=cleaned_data['person'],
        contract_period=cleaned_data['contract_term'],
        **as_is(
            'employment_type',
            'salary',
            'currency',
            'other_payments',
            'join_at',
            'is_replacement',
            'contract_term_date',
        ),
        **resolve_id(
            'office',
            'department',
            'organization',
        )
    )
    service = WorkflowRegistryService()
    try:
        workflow_id = service.try_create_workflow_from_femida(femida_data)
    except OebsHireError as e:
        response = AssignmentCreationFailedResponse(e.oebs_internal_errors)
        return JsonResponse(attr.asdict(response), status=500)

    response = AssignmentCreateResponse(str(workflow_id))
    return JsonResponse(attr.asdict(response), status=201)
