from lagom import Container, Singleton, dependency_definition
import waffle

from staff.budget_position.workflow_service import entities, gateways, use_cases
from staff.budget_position.tasks import update_push_status


def build_container() -> Container:
    workflow_service_container = Container()
    workflow_service_container[entities.WorkflowRepositoryInterface] = Singleton(gateways.WorkflowRepository)
    workflow_service_container[entities.BudgetPositionsRepository] = Singleton(gateways.BudgetPositionsRepository)
    workflow_service_container[entities.StaffService] = Singleton(gateways.StaffService)
    workflow_service_container[entities.OEBSService] = Singleton(gateways.OEBSService)
    workflow_service_container[use_cases.OebsHireService] = Singleton(gateways.OebsHireService)
    workflow_service_container[use_cases.CreditManagementServiceInterface] = Singleton(gateways.CreditManagementService)
    workflow_service_container[entities.FemidaService] = Singleton(gateways.FemidaService)
    workflow_service_container[entities.TableflowService] = Singleton(gateways.TableflowService)
    workflow_service_container[entities.AbstractProposalWorkflowFactory] = entities.ProposalWorkflowFactory

    @dependency_definition(workflow_service_container)
    def create_push_workflow(c: Container) -> use_cases.PushWorkflowToOebs:
        staff_service = c[entities.StaffService]
        oebs_service = c[entities.OEBSService]
        workflows_repository = c[entities.WorkflowRepositoryInterface]
        return use_cases.PushWorkflowToOebs(workflows_repository, oebs_service, staff_service, update_push_status)

    @dependency_definition(workflow_service_container)
    def create_update_workflow_push_status(c: Container) -> use_cases.UpdateWorkflowPushStatus:
        staff_service = c[entities.StaffService]
        oebs_service = c[entities.OEBSService]
        workflows_repository = c[entities.WorkflowRepositoryInterface]
        credit_management_service = c[use_cases.CreditManagementServiceInterface]
        return use_cases.UpdateWorkflowPushStatus(
            workflows_repository,
            oebs_service,
            update_push_status,
            credit_management_service,
            staff_service,
        )

    @dependency_definition(workflow_service_container)
    def workflow_factory(c: Container) -> entities.AbstractFemidaWorkflowFactory:
        staff_service = c[entities.StaffService]
        oebs_service = c[entities.OEBSService]
        budget_positions_repo = c[entities.BudgetPositionsRepository]

        if waffle.switch_is_active('enable_bp_state_checks_in_registry'):
            return entities.FemidaWorkflowFactory(staff_service, oebs_service, budget_positions_repo)

        return entities.FemidaWorkflowFactoryWithoutChecks(staff_service, oebs_service, budget_positions_repo)

    return workflow_service_container
