import datetime
from decimal import Decimal
from typing import Set
from uuid import UUID

import attr

from staff.map.models import Placement
from staff.budget_position.const import PositionType
from staff.budget_position.converters import (
    date_converter,
    rate_converter,
    salary_converter,
    pay_system_converter,
    position_name_converter,
    wage_system_converter,
)
from staff.budget_position.workflow_service.entities.budget_positon import BudgetPosition


@attr.s(cmp=False, auto_attribs=True)
class Change:
    id: int or None = None
    workflow_id: UUID or None = None
    budget_position: BudgetPosition or None = None
    new_budget_position: BudgetPosition or None = None
    linked_budget_position_id: int or None = None

    pushed_to_femida: datetime.datetime or None = None
    oebs_transaction_id: int or None = None
    oebs_idempotence_key: UUID or None = None
    last_oebs_error: str or None = None
    correction_id: int or None = None
    push_status: str or None = None
    sent_to_oebs: datetime.datetime or None = None
    assignment_id: int or None = None

    currency: str or None = None
    department_id: int or None = None
    dismissal_date: datetime.date or None = attr.ib(default=None, converter=date_converter)
    effective_date: datetime.date or None = attr.ib(default=None, converter=date_converter)
    geography_url: str or None = None
    grade_id: int or None = None  # grade_ID in oebs terms
    headcount: int or None = None
    hr_product_id: int or None = None
    office_id: int or None = None
    organization_id: int or None = None
    pay_system: str or None = attr.ib(default=None, converter=pay_system_converter)
    wage_system: str or None = attr.ib(default=None, converter=wage_system_converter)
    placement_id: int or None = None
    position_id: int or None = None
    position_name: str or None = attr.ib(default=None, converter=position_name_converter)
    position_type: PositionType or None = None
    rate: Decimal or None = attr.ib(default=None, converter=rate_converter)
    remove_budget_position: bool or None = None
    salary: Decimal or None = attr.ib(default=None, converter=salary_converter)
    ticket: str or None = None
    optional_ticket: str or None = None
    unit_manager: bool or None = None

    review_scheme_id: int or None = None
    bonus_scheme_id: int or None = None
    reward_scheme_id: int or None = None  # reward

    employment_type: str or None = None
    person_id: int or None = None
    physical_entity_id: int or None = None
    other_payments: str or None = None
    join_at: datetime.date or None = attr.ib(default=None, converter=date_converter)
    probation_period_code: int or None = None
    is_replacement: bool or None = None
    instead_of_login: str or None = None
    contract_term_date: datetime.date or None = attr.ib(default=None, converter=date_converter)
    contract_period: int = None

    @property
    def changes_existing_budget_position(self) -> bool:
        return bool(self.budget_position) and not self.new_budget_position

    @property
    def creates_new_budget_positions(self) -> bool:
        return not self.new_budget_position and not self.budget_position and self.position_name

    @property
    def moves_from_one_budget_position_to_another(self) -> bool:
        if self.budget_position and self.new_budget_position:
            return self.budget_position != self.new_budget_position

        return False

    def fields_with_values(self) -> Set[str]:
        all_fields = [field.name for field in attr.fields(Change)]
        result = {
            field
            for field in all_fields
            if getattr(self, field, None)
        }

        return result

    def __attrs_post_init__(self):
        if not self.placement_id and self.organization_id and self.office_id:
            placement = (
                Placement.objects
                .filter(
                    organization_id=self.organization_id,
                    office_id=self.office_id,
                    active_status=True,
                )
                .first()
            )
            self.placement_id = placement and placement.id

    def squash(self, other: 'Change') -> 'Change':
        values = {}
        for field in attr.fields(self.__class__):
            values[field.name] = getattr(self, field.name) or getattr(other, field.name)
        return Change(**values)
