from datetime import date
from decimal import Decimal
import logging
from typing import List, Optional
import uuid

import attr
from staff.lib import waffle

from staff.budget_position.const import PositionType
from staff.budget_position.models import BudgetPositionAssignment, BudgetPositionAssignmentStatus
from staff.budget_position.workflow_service.entities.abstract_workflow import AbstractWorkflow
from staff.budget_position.workflow_service.entities.abstract_workflow_factory import AbstractFemidaWorkflowFactory
from staff.budget_position.workflow_service.entities.change import Change
from staff.budget_position.workflow_service.entities.interfaces import (
    BudgetPositionsRepository,
    OEBSService,
    SalaryData,
    StaffService,
)
from staff.budget_position.workflow_service.entities.service_dtos import FemidaData, OccupationDetails
from staff.budget_position.workflow_service.entities import (
    errors,
    workflows,
    BudgetPositionCode,
    UnexpectedResponseOEBSError,
)

logger = logging.getLogger(__name__)


class FemidaWorkflowFactoryWithoutChecks(AbstractFemidaWorkflowFactory):
    def __init__(
        self,
        staff_service: StaffService,
        oebs_service: OEBSService,
        budget_positions_repository: BudgetPositionsRepository,
    ) -> None:
        super().__init__()
        self._staff_service = staff_service
        self._oebs_service = oebs_service
        self._budget_positions_repository = budget_positions_repository

    def create(self, data: FemidaData) -> AbstractWorkflow:
        if data.is_assignment_creation:
            change = self._create_change(data, None)
            result = workflows.Workflow5_1_1(uuid.uuid1(), [change])
            return result

        if data.budget_position_code is None:
            if data.is_vacancy:
                change = self._create_change(data, PositionType.VACANCY)
                changes = self._make_changes_with_credit([change])
                result = workflows.Workflow1_2(uuid.uuid1(), changes)
                return result
            else:
                logger.info('Empty BP was passed, but this is not a vacancy creation')
                raise errors.EmptyBPOnNonVacancyCreation('Empty BP was passed, but this is not a vacancy creation')

        if not self._changes_existing_budget_position(data):
            # Существование БП должен был проверить вызывающий код
            logger.error('Non existent BP was passed')
            raise NotImplementedError('Workflow for conditions not found')

        logger.info('About to change existing budget position')

        if data.is_vacancy_cancellation:
            change = self._create_change(data, PositionType.EMPTY)
            result = workflows.Workflow1_3(uuid.uuid1(), [change])
            return result

        if data.is_vacancy:
            assert data.budget_position_code is not None
            if self._changes_vacancy_plan_budget_position(data):
                change = self._create_change(data, PositionType.VACANCY)
                result = workflows.Workflow1_1(uuid.uuid1(), [change])
                return result
            else:
                data.grade_id = self._get_grade_id(data.budget_position_code)
                logger.info('Took oebs grade_id %s for budget position %s', data.grade_id, data.budget_position_code)
                change = self._create_change(data, PositionType.VACANCY)
                result = workflows.Workflow2_1(uuid.uuid1(), [change])
                return result

        if data.is_offer:
            change = self._create_change(data, PositionType.OFFER)
            result = workflows.Workflow5_1(uuid.uuid1(), [change])
            return result

        if data.is_offer_rejection:
            change = self._create_change(data, PositionType.VACANCY)
            result = workflows.Workflow5_2(uuid.uuid1(), [change])
            return result

        if data.is_internal_offer:
            change = self._create_change(data, PositionType.OFFER)
            result = workflows.Workflow5_3(uuid.uuid1(), [change])
            self._update_changes_for_internal_offer(result)
            return result

        logger.error('Can\'t detect action, probably budget position is not synced with OEBS')
        raise errors.NotSupportedBPState(data.budget_position_code, 'Not supported BP state ')

    def _get_grade_id(self, budget_position_code: BudgetPositionCode) -> int:
        grade_id = self._oebs_service.get_position_as_change(budget_position_code).grade_id
        if not self._oebs_service.get_grade_data(grade_id):
            raise UnexpectedResponseOEBSError('OEBS returned unknown grade id')
        return grade_id

    def _calculate_hr_product(self, income_hr_product: int, budget_position_code: BudgetPositionCode) -> int:
        if income_hr_product:
            return income_hr_product

        if not budget_position_code:
            return income_hr_product

        assignment: Optional[BudgetPositionAssignment] = (
            BudgetPositionAssignment.objects
            .active()
            .filter(budget_position__code=budget_position_code, next_assignment=None)
            .first()
        )
        if assignment is None:
            raise errors.UnknownAssignment('No corresponding assignment exists')
        result = assignment.value_stream.hrproduct_set.first()
        if result is None:
            raise errors.UnknownHrProduct('VS for assignment has no HR product')
        return result.id

    def _create_change(self, data: FemidaData, position_type: PositionType or None) -> Change:
        budget_position = self._budget_positions_repository.budget_position_by_code_or_none(data.budget_position_code)
        hr_product_id = self._calculate_hr_product(data.hr_product_id, data.budget_position_code)

        result = Change(
            bonus_scheme_id=data.bonus_scheme_id,
            budget_position=budget_position,
            contract_period=data.contract_period,
            contract_term_date=data.contract_term_date,
            currency=data.currency,
            department_id=data.department_id,
            dismissal_date=data.dismissal_date,
            effective_date=date.today(),
            employment_type=data.employment_type,
            geography_url=data.geography_url,
            grade_id=data.grade_id,
            hr_product_id=hr_product_id,
            instead_of_login=data.instead_of_login,
            is_replacement=data.is_replacement,
            join_at=data.join_at,
            office_id=data.office_id,
            organization_id=data.organization_id,
            other_payments=data.other_payments,
            pay_system=data.payment_system,
            person_id=data.person_id,
            physical_entity_id=data.physical_entity_id,
            position_id=data.position_id,
            position_type=position_type,
            probation_period_code=data.probation_period_code,
            rate=data.rate,
            review_scheme_id=data.review_scheme_id,
            reward_scheme_id=data.reward_scheme_id,
            salary=data.salary and Decimal(data.salary),
            ticket=data.ticket,
            optional_ticket=data.salary_ticket,
            wage_system=data.wage_system,
        )

        return result

    def _make_changes_with_credit(self, changes: List[Change]) -> List[Change]:
        assert len(changes) == 1

        position_name = 'Вакансия'
        if changes[0].grade_id:
            grade_data = self._oebs_service.get_grade_data(changes[0].grade_id)
            assert grade_data is not None
            occupation_details: OccupationDetails = self._staff_service.occupation_details(
                grade_data.occupation
            )
            assert occupation_details is not None
            position_name = occupation_details.occupation_description
            assert position_name

        change = attr.evolve(
            changes[0],
            position_name=position_name,
            unit_manager=False,
            headcount=1,
            rate=Decimal(1),
        )

        result = attr.evolve(
            change,
            position_type=PositionType.NEGATIVE_RESERVE,
            position_name=f'Кредит для {change.ticket}',
        )

        return [result, change]

    def _changes_existing_budget_position(self, data: FemidaData) -> bool:
        return data.budget_position_code is not None

    def _changes_vacancy_plan_budget_position(self, data: FemidaData) -> bool:
        assert self._changes_existing_budget_position(data)
        statuses = self._budget_positions_repository.assignments_statuses_by_codes([data.budget_position_code])
        return statuses == {BudgetPositionAssignmentStatus.VACANCY_PLAN.value}

    def _get_salary_data(self, person_login: str):
        if waffle.switch_is_active('disable_salary_request'):
            return SalaryData(salary=Decimal(), rate=Decimal())

        return self._oebs_service.get_salary_data(person_login)

    def _update_changes_for_internal_offer(self, workflow):
        assert len(workflow.changes) == 1
        assert workflow.changes[0].person_id is not None
        change = workflow.changes[0]

        person = self._staff_service.get_person(change.person_id)
        office_id = person.office_id
        organization_id = person.organization_id

        if not change.office_id or not change.organization_id:
            office_id = change.office_id or office_id
            organization_id = change.organization_id or organization_id

            placement = self._staff_service.placement_for(office_id, organization_id)
            change.placement_id = placement and placement.id
            change.office_id = office_id
            change.organization_id = organization_id

        if not change.salary or not change.rate:
            salary_data = self._get_salary_data(person.login)
            change.salary = change.salary or salary_data.salary
            change.rate = change.rate or salary_data.rate
