import logging
from typing import List, Dict, Tuple

import attr

from staff.budget_position.const import FemidaProfessionalLevel
from staff.budget_position.workflow_service.entities.interfaces import OEBSService, GradeData, StaffService
from staff.budget_position.workflow_service.entities.types import GradeId, OccupationId, PersonId


logger = logging.getLogger(__name__)


@attr.s(auto_attribs=True)
class GradeChange:
    new_occupation_id: OccupationId
    grade_level_change: int

    @property
    def has_occupation_change(self) -> bool:
        return bool(self.new_occupation_id)

    @property
    def has_grade_level_change(self) -> bool:
        return bool(self.grade_level_change)

    @property
    def changes_grade(self) -> bool:
        return self.has_grade_level_change or self.has_occupation_change


@attr.s(auto_attribs=True)
class CalculatedGrade:
    grade_id: int
    grade_level: int


class GradeCalculator:
    def __init__(self, oebs_service: OEBSService, staff_service: StaffService):
        self._staff_service = staff_service
        self._oebs_service = oebs_service
        self._grades_data_cache: Dict[int, GradeData] = {}

    def current_grades_data(self, person_ids: List[PersonId]) -> Dict[PersonId, GradeData]:
        """For list of person ids return map of person id to current grade data"""
        result = {
            person_id: self._grades_data_cache[person_id]
            for person_id in person_ids
            if person_id in self._grades_data_cache
        }

        person_ids = [person_id for person_id in person_ids if person_id not in result]
        if person_ids:
            logins = self._staff_service.get_person_logins(person_ids)
            logger.info('Requesting current grade data for %s', logins)
            grade_data = self._oebs_service.get_grades_data(list(logins.values()))
            login_to_id = {login: id for id, login in logins.items()}
            new_data = {login_to_id[login]: grade for login, grade in grade_data.items()}
            logger.info('OEBS returned grades for %s', grade_data.keys())
            result.update(new_data)
            self._grades_data_cache.update(new_data)

        return result

    def calculate_new_grades(self, changes: Dict[PersonId, List[GradeChange]]) -> Dict[PersonId, List[CalculatedGrade]]:
        """
        For map of person ids to new occupation and grade delta (not absolute value) returns
        map of person ids to new grade id (grade id in terms of oebs)
        """
        result = {}
        current_grades_data = self.current_grades_data(list(changes.keys()))
        for person_id, current_grade_data in current_grades_data.items():
            if current_grade_data is None:
                result[person_id] = []
                continue

            person_grade_changes = changes[person_id]

            current_occupation = current_grade_data.occupation
            current_level = current_grade_data.level

            person_result = []
            for change in person_grade_changes:
                if change.changes_grade:
                    current_occupation = (
                        change.new_occupation_id
                        if change.has_occupation_change
                        else current_occupation
                    )

                    current_level = (
                        current_level + change.grade_level_change
                        if change.has_grade_level_change and current_level is not None
                        else current_level
                    )

                    grade_id = self._oebs_service.get_grade_id(current_occupation, current_level)
                    person_result.append(CalculatedGrade(grade_id=grade_id, grade_level=current_level))
                else:
                    person_result.append(None)

            result[person_id] = person_result

        return result

    def grade_by_prof_level(
        self,
        occupation_id: OccupationId,
        prof_level: FemidaProfessionalLevel,
    ) -> Tuple[GradeId, int]:
        grade_level = self._oebs_service.get_grade_id_by_femida_level(occupation_id, prof_level)
        grade_id = self._oebs_service.get_grade_id(occupation_id, grade_level)
        return grade_id, grade_level
