from datetime import date, timedelta

from staff.budget_position.workflow_service.entities.service_dtos import RewardSchemeDetails, GetSchemeRequest


class InsuranceCalculator:
    SIX_MONTHS = 6
    SIX_MONTHS_IN_DAYS = 185

    def __init__(self, reward_scheme_details: RewardSchemeDetails, scheme_request: GetSchemeRequest):
        self._reward_scheme_details = reward_scheme_details
        self._scheme_request = scheme_request

    def has_health_insurance(self) -> bool:
        if not self._has_health_insurance_option_in_scheme():
            return False

        if not self._scheme_request.is_main_work_place:
            return False

        return self._will_work_more_than_six_months()

    def has_life_insurance(self) -> bool:
        if not self._has_life_insurance_option_in_scheme():
            return False

        if not self._scheme_request.is_main_work_place:
            return False

        if self._has_health_insurance_option_in_scheme():
            return self._will_work_more_than_six_months()

        return True

    def _has_life_insurance_option_in_scheme(self) -> bool:
        return any(
            details
            for details in self._reward_scheme_details.ai
            if details.name != 'Без НС'
        )

    def _has_health_insurance_option_in_scheme(self) -> bool:
        return any(
            details
            for details in self._reward_scheme_details.dms
            if details.name != 'Без ДМС'
        )

    def _will_work_more_than_six_months(self) -> bool:
        if self._is_months_term_contract():
            return self._scheme_request.contract_term > self.SIX_MONTHS

        if self._is_till_date_term_contract():
            term: timedelta = self._scheme_request.contract_term_date - date.today()
            return term.days > self.SIX_MONTHS_IN_DAYS

        return True

    def _is_months_term_contract(self) -> bool:
        return self._scheme_request.contract_term is not None

    def _is_till_date_term_contract(self) -> bool:
        return self._scheme_request.contract_term_date is not None
