from abc import ABCMeta, abstractmethod
from typing import List, Dict, Optional

import attr

from staff.budget_position.workflow_service.entities.service_dtos import (
    BonusSchemeDetails,
    ReviewSchemeDetails,
    RewardSchemeDetails,
    OccupationDetails,
)
from staff.budget_position.workflow_service.entities.types import (
    BonusSchemeId,
    OccupationId,
    PersonId,
    ReviewSchemeId,
    RewardSchemeId,
)


@attr.s(auto_attribs=True, hash=True)
class Person:
    id: PersonId
    login: str
    office_id: int
    organization_id: Optional[int]


@attr.s(auto_attribs=True)
class Placement:
    id: int
    office_id: int
    organization_id: int


@attr.s(auto_attribs=True)
class PersonSchemeException:
    person_id: PersonId
    bonus_scheme_id: int
    review_scheme_id: int
    reward_scheme_id: int


class StaffService(metaclass=ABCMeta):
    @abstractmethod
    def hr_analyst_responsible_for_department(self, department_id) -> Optional[Person]:
        pass

    @abstractmethod
    def default_hr_analyst(self) -> Person:
        pass

    @abstractmethod
    def get_person(self, person_id: PersonId) -> Person:
        pass

    @abstractmethod
    def placement_for(self, office_id: int, organization_id: int) -> Optional[Placement]:
        pass

    @abstractmethod
    def bonus_scheme_details(self, bonus_scheme_id: BonusSchemeId) -> Optional[BonusSchemeDetails]:
        pass

    @abstractmethod
    def review_scheme_details(self, review_scheme_id: ReviewSchemeId) -> Optional[ReviewSchemeDetails]:
        pass

    @abstractmethod
    def reward_scheme_details(self, reward_scheme_id: RewardSchemeId) -> Optional[RewardSchemeDetails]:
        pass

    @abstractmethod
    def occupation_details(self, occupation_id: OccupationId) -> Optional[OccupationDetails]:
        pass

    @abstractmethod
    def multiple_occupation_groups(
        self,
        occupation_ids: List[OccupationId],
    ) -> Dict[OccupationId, Optional[OccupationDetails]]:
        pass

    @abstractmethod
    def get_person_logins(self, person_ids: List[PersonId]) -> Dict[PersonId, str]:
        pass

    @abstractmethod
    def person_departments(self, person_ids: List[PersonId]) -> Dict[PersonId, int]:
        pass

    @abstractmethod
    def person_scheme_exceptions(self, person_ids: List[PersonId]) -> Dict[PersonId, PersonSchemeException]:
        pass
