from abc import ABCMeta, abstractmethod
from typing import List, Optional
from uuid import UUID

from staff.person.models import Staff

from staff.budget_position.workflow_service.entities import AbstractWorkflow


class WorkflowRepositoryInterface(metaclass=ABCMeta):
    @abstractmethod
    def save(self, workflow: AbstractWorkflow) -> UUID:
        pass

    @abstractmethod
    def get_pending_workflows_by_budget_position_code(self, budget_position_code: int):
        pass

    @abstractmethod
    def get_by_id(self, workflow_id: UUID) -> AbstractWorkflow:
        pass

    @abstractmethod
    def get_workflows_by_proposal_id(self, proposal_id: int, status: str) -> List[AbstractWorkflow]:
        pass

    @abstractmethod
    def get_workflows_by_credit_management_application(self, credit_repayment_id: int) -> List[AbstractWorkflow]:
        pass

    @abstractmethod
    def cancel_workflows_for_credit_repayment(self, credit_repayment_id: int) -> None:
        pass

    @abstractmethod
    def is_all_workflows_for_credit_repayment_not_in_pending(self, credit_management_id: int) -> bool:
        pass

    @abstractmethod
    def is_all_workflows_for_credit_repayment_is_cancelled(self, credit_management_id: int) -> bool:
        pass

    @abstractmethod
    def queue_workflows(self, workflow_ids: List[UUID], person: Optional[Staff]) -> None:
        pass

    @abstractmethod
    def retry_workflows(self, workflow_ids: List[UUID], person: Staff) -> None:
        pass

    @abstractmethod
    def get_workflow_list(self, workflow_ids: List[UUID]) -> List[AbstractWorkflow]:
        pass

    @abstractmethod
    def get_workflows(self, status: str) -> List[UUID]:
        pass

    @abstractmethod
    def can_workflow_be_finalized(self, workflow_id: UUID) -> bool:
        pass

    @abstractmethod
    def get_related_department_ids(self, workflow_id: UUID) -> List[int]:
        pass

    @abstractmethod
    def get_related_tickets(self, workflow_id: UUID) -> List[str]:
        pass

    @abstractmethod
    def mark_changes_as_failed(self, workflow_id: UUID, exc: Exception) -> None:
        pass

    @abstractmethod
    def cancel_workflows_for_proposal(self, proposal_id: str):
        pass
