from decimal import Decimal
from datetime import date
from functools import reduce
import logging
import uuid
from typing import List, Optional

import attr
from staff.lib import waffle

from staff.budget_position.const import PositionType
from staff.budget_position.workflow_service.entities.abstract_workflow import AbstractWorkflow
from staff.budget_position.workflow_service.entities.abstract_workflow_factory import AbstractProposalWorkflowFactory
from staff.budget_position.workflow_service.entities.budget_positions_calculator import BudgetPositionMove
from staff.budget_position.workflow_service.entities.change import Change
from staff.budget_position.workflow_service.entities.interfaces import (
    OEBSService,
    StaffService,
)
from staff.budget_position.workflow_service.entities.service_dtos import ProposalData
from staff.budget_position.workflow_service.entities import workflows

logger = logging.getLogger(__name__)


class ProposalWorkflowFactory(AbstractProposalWorkflowFactory):
    def __init__(
        self,
        staff_service: StaffService,
        oebs_service: OEBSService,
    ) -> None:
        super().__init__()
        self._staff_service = staff_service
        self._oebs_service = oebs_service

    def create_workflow(self, data: ProposalData, budget_position_move: BudgetPositionMove) -> AbstractWorkflow:
        changes = self._create_changes(data, budget_position_move)

        if data.has_headcount_position_code:
            logger.info(
                'Changing free headcount, choosing %s',
                workflows.HeadcountChangeFromProposalWorkflow.code,
            )
            result = workflows.HeadcountChangeFromProposalWorkflow(uuid.uuid1(), changes)
            return result

        if data.has_vacancy_id:
            logger.info('Changing vacancy, choosing WF 1.1.1')
            result = workflows.Workflow1_1_1(uuid.uuid1(), changes)
            return result

        if data.has_person_id:
            if not data.is_move_with_budget_position:
                logger.info('Move without budget position, choosing WF 7.3')
                return self._create_move_without_budget_position_workflow(changes)

            if not budget_position_move.has_any_budget_position:
                logger.info(
                    'Using fallback workflow 7.100500, provided person_id %s', data.person_id,
                )
                return self._create_workflow_7_100500(changes)

            logger.info('Using workflow 7.1')
            return self._create_workflow_7_1(changes)

        logger.error(
            'Can\'t detect workflow, you have to provide either'
            ' headcount_position_code, vacancy_id, person_id'
        )
        raise NotImplementedError('Workflow for conditions not found')

    def _create_changes(
        self,
        data: ProposalData,
        budget_position_move: BudgetPositionMove,
    ) -> List[Change]:
        result = []
        for requested_change in data.proposal_changes:
            change = Change(
                office_id=requested_change.office_id,
                department_id=requested_change.department_id,
                organization_id=requested_change.organization_id,
                grade_id=requested_change.grade_id,
                hr_product_id=requested_change.hr_product_id,
                geography_url=requested_change.geography_url,
                salary=requested_change.salary and Decimal(requested_change.salary),
                currency=requested_change.currency,
                rate=requested_change.rate,
                wage_system=requested_change.wage_system,
                pay_system=requested_change.wage_system,
                position_id=requested_change.position_id,
                effective_date=requested_change.oebs_date,
                ticket=data.ticket,
                person_id=data.person_id,
                review_scheme_id=requested_change.review_scheme_id,
                bonus_scheme_id=requested_change.bonus_scheme_id,
                reward_scheme_id=requested_change.reward_scheme_id,
            )
            if budget_position_move.budget_position_will_be_changed:
                change.new_budget_position = budget_position_move.new_budget_position
                change.budget_position = budget_position_move.old_budget_position
            else:
                change.budget_position = budget_position_move.new_budget_position

            result.append(change)

        assert len(result) > 0
        return result

    def _grade_id_for_person(self, login: str) -> Optional[int]:
        grade_data = self._oebs_service.get_grades_data([login])[login]
        if not grade_data:
            logger.info('OEBS returned empty grade data for %s', login)
            return None

        grade_id = self._oebs_service.get_grade_id(grade_data.occupation, grade_data.level)
        if grade_id is None:
            logger.info('grade_id was not found for %s %s', grade_data.occupation, grade_data.level)
        return grade_id

    def _apply_info_from_oebs_for_move_without_bp(self, change: Change) -> Change:
        login = self._staff_service.get_person_logins([change.person_id])[change.person_id]

        currency = change.currency
        salary = change.salary
        rate = change.rate

        if not all((currency, salary, rate)):
            salary_data = self._oebs_service.get_salary_data(login)
            if not currency:
                currency = salary_data.currency
            if not salary:
                salary = salary_data.salary
            if not rate:
                rate = salary_data.rate

        placement_id = change.placement_id
        organization_id = change.organization_id

        if not all((placement_id, organization_id)):
            oebs_info = self._oebs_service.get_position_as_change(change.budget_position.code)
            if not placement_id:
                placement_id = oebs_info.placement_id
            if not organization_id:
                organization_id = oebs_info.organization_id

        change_with_oebs_info = attr.evolve(
            change,
            currency=currency,
            salary=salary,
            rate=rate,
            grade_id=change.grade_id or self._grade_id_for_person(login),
            placement_id=placement_id,
            organization_id=organization_id,
        )

        return change_with_oebs_info

    def _create_move_without_budget_position_workflow(self, changes: List[Change]) -> AbstractWorkflow:
        change = reduce(
            lambda first, second: first.squash(second),
            sorted(changes, key=lambda ch: ch.effective_date),
        )
        self._calculate_placement_and_update_changes([change])
        change_with_oebs_info = (
            self._apply_info_from_oebs_for_move_without_bp(change)
            if waffle.switch_is_active('copy_oebs_info_when_move_without_budget')
            else change
        )
        change_with_oebs_info.position_type = PositionType.OFFER  # STAFF-16770
        change_with_oebs_info.effective_date = date.today()

        result = workflows.MoveWithoutBudgetPositionWorkflow(uuid.uuid1(), [change_with_oebs_info])
        return result

    def _create_workflow_7_1(self, changes: List[Change]) -> AbstractWorkflow:
        result = workflows.Workflow7_1(uuid.uuid1(), changes)
        self._calculate_placement_and_update_changes(result.changes_by_effective_date)
        return result

    def _create_workflow_7_100500(self, changes: List[Change]) -> AbstractWorkflow:
        result = workflows.Workflow7_100500(uuid.uuid1(), changes)
        self._calculate_placement_and_update_changes(result.changes_by_effective_date)
        return result

    def _calculate_placement_and_update_changes(self, changes: List[Change]):
        assert len(changes) != 0
        if changes[0].person_id is None:  # change for vacancy
            return

        person = self._staff_service.get_person(changes[0].person_id)
        prev_office = person.office_id
        prev_organization = person.organization_id

        for change in changes:
            if change.office_id is None and change.organization_id is None:
                continue

            considered_office = prev_office if change.office_id is None else change.office_id
            considered_organization = prev_organization if change.organization_id is None else change.organization_id

            placement = self._staff_service.placement_for(considered_office, considered_organization)
            change.placement_id = placement and placement.id

            if not placement:
                logger.info(
                    'Can\'t find placement for office %s and organization %s',
                    considered_office,
                    considered_organization,
                )

            prev_office, prev_organization = considered_office, considered_organization
