import logging
from typing import Optional, List, Dict

import attr
from staff.lib import waffle

from staff.budget_position.workflow_service.entities.abstract_workflow import AbstractWorkflow
from staff.budget_position.workflow_service.entities.change import Change
from staff.budget_position.workflow_service.entities.grade_calculator import GradeCalculator
from staff.budget_position.workflow_service.entities.interfaces import (
    TableflowService,
    StaffService,
    PersonSchemeException,
    ReviewSchemeIdRequest,
    ReviewSchemeIdByGroupRequest,
    RewardCategoryRequest,
    RewardSchemeIdRequest,
    BonusSchemeIdRequest,
    BonusSchemeIdByGroupRequest,
)
from staff.budget_position.workflow_service.entities.scheme_requests_collector import (
    SchemeRequestsCollector,
    ChangeSchemeRequest,
)
from staff.budget_position.workflow_service.entities.service_dtos import (
    BonusSchemeDetails,
    FemidaData,
    OccupationDetails,
    ReviewSchemeDetails,
    RewardSchemeDetails,
    MONTHS_PER_YEAR,
)
from staff.budget_position.workflow_service.entities.workflows import (
    Workflow1_1,
    Workflow1_2,
    Workflow2_1,
    Workflow5_1,
    Workflow5_3,
    Workflow7_1,
)

from staff.budget_position.workflow_service.entities.types import (
    BonusSchemeId,
    OccupationId,
    PersonId,
    RewardSchemeId,
    ReviewSchemeId,
)


logger = logging.getLogger(__name__)


@attr.s(auto_attribs=True)
class SchemesSet:
    """
    Результат работы, отдается в паре с ChangeSchemeRequest
    """
    review_scheme_id: Optional[ReviewSchemeId]
    bonus_scheme_id: Optional[BonusSchemeId]
    reward_scheme_id: Optional[RewardSchemeId]


class SchemesCalculator:
    """
    Высчитывает схемы по БП если есть изменение грейда
    """
    def __init__(self, table_flow: TableflowService, staff_service: StaffService, grade_calculator: GradeCalculator):
        self._table_flow = table_flow
        self._staff_service = staff_service
        self._grade_calculator = grade_calculator

    def bonus_scheme_details(self, bonus_scheme_id: BonusSchemeId) -> Optional[BonusSchemeDetails]:
        return self._staff_service.bonus_scheme_details(bonus_scheme_id)

    @staticmethod
    def _is_non_review_bonus_row(row):
        return row.value_type == 'Процент от оклада' and row.value_source == 'Значением'

    def review_scheme_details(self, review_scheme_id: ReviewSchemeId) -> Optional[ReviewSchemeDetails]:
        result = self._staff_service.review_scheme_details(review_scheme_id)
        if not result:
            return result

        result.has_review = result.name != 'DEFAULT'
        result.review_bonus = (result.target_bonus or 0.0) * MONTHS_PER_YEAR if result.has_review else None
        return result

    def reward_scheme_details(self, reward_scheme_id: RewardSchemeId) -> RewardSchemeDetails or None:
        result = self._staff_service.reward_scheme_details(reward_scheme_id)
        return result

    def scheme_set(
        self,
        department_id: Optional[int],
        occupation_id: Optional[OccupationId],
        grade_level: Optional[int],
        is_internship: Optional[bool],
    ) -> Optional[SchemesSet]:
        if department_id is None or occupation_id is None or grade_level is None or is_internship is None:
            logger.info('Will not search for schemes, not enough input data')
            return None

        occupation_details = self._staff_service.occupation_details(occupation_id)
        if not occupation_details:
            logger.info('Will not search for schemes, cant found occupation details for occupation %s', occupation_id)

        review_scheme_id = self.review_scheme_id(
            department_id=department_id,
            occupation_id=occupation_id,
            review_group=occupation_details.review_group,
            grade_level=grade_level,
        )
        bonus_scheme_id = self.bonus_scheme_id(
            bonus_group=occupation_details.bonus_group,
            grade_level=grade_level,
            department_id=department_id,
            occupation_id=occupation_id,
        )
        reward_scheme_id = self.reward_scheme_id(
            reward_group=occupation_details.reward_group,
            grade_level=grade_level,
            department_id=department_id,
            is_internship=is_internship,
        )
        return SchemesSet(review_scheme_id, bonus_scheme_id, reward_scheme_id)

    def schemes_set_for_mass_person_changes(
        self,
        requests: List[ChangeSchemeRequest],
    ) -> Dict[ChangeSchemeRequest, SchemesSet]:
        if not waffle.switch_is_active('enable_table_flow_for_registry'):
            return {}

        scheme_requests_collector = SchemeRequestsCollector(requests)
        if not scheme_requests_collector.has_any_change_that_leads_to_scheme_recalculation():
            logger.info('There is no changes that will change schemes')
            return {}

        involved_staff_ids = scheme_requests_collector.involved_staff_ids()
        assert involved_staff_ids

        person_exceptions = self._staff_service.person_scheme_exceptions(involved_staff_ids)
        scheme_requests_collector.exclude_persons_from_further_calculations(list(person_exceptions.keys()))

        if not scheme_requests_collector.involved_staff_ids():
            return self._make_result(person_exceptions, {}, {}, {}, requests)

        person_departments = self._staff_service.person_departments(involved_staff_ids)
        grades_data = self._grade_calculator.current_grades_data(involved_staff_ids)

        scheme_requests_collector.prepare_data_for_request_by_occupation(person_departments, grades_data)
        occupation_groups = self._staff_service.multiple_occupation_groups(
            list(scheme_requests_collector.involved_occupations)
        )

        review_scheme_ids = self._review_scheme_ids(scheme_requests_collector, occupation_groups)
        bonus_scheme_ids = self._bonus_scheme_ids(scheme_requests_collector, occupation_groups)
        reward_scheme_ids = self._reward_schmes_ids(scheme_requests_collector, occupation_groups)
        return self._make_result(person_exceptions, review_scheme_ids, bonus_scheme_ids, reward_scheme_ids, requests)

    def _make_result(
        self,
        person_exceptions: Dict[PersonId, PersonSchemeException],
        review_scheme_ids: Dict[ChangeSchemeRequest, ReviewSchemeId],
        bonus_scheme_ids: Dict[ChangeSchemeRequest, BonusSchemeId],
        reward_scheme_ids: Dict[ChangeSchemeRequest, RewardSchemeId],
        requests: List[ChangeSchemeRequest],
    ):
        result = {}
        for request in requests:
            if request.person_id in person_exceptions:
                person_exception = person_exceptions[request.person_id]
                result[request] = SchemesSet(
                    review_scheme_id=person_exception.review_scheme_id,
                    bonus_scheme_id=person_exception.bonus_scheme_id,
                    reward_scheme_id=person_exception.reward_scheme_id,
                )
            else:
                result[request] = SchemesSet(
                    review_scheme_id=review_scheme_ids.get(request),
                    bonus_scheme_id=bonus_scheme_ids.get(request),
                    reward_scheme_id=reward_scheme_ids.get(request),
                )

        return result

    def _reward_schmes_ids(
        self,
        scheme_requests_collector: SchemeRequestsCollector,
        occupation_groups: Dict[OccupationId, OccupationDetails or None],
    ) -> Dict[ChangeSchemeRequest, RewardSchemeId]:
        categories = self._request_reward_categories_by_occupation_groups(scheme_requests_collector, occupation_groups)
        reward_scheme_ids = self._request_reward_scheme_ids_by_categories(scheme_requests_collector, categories)
        return dict(reward_scheme_ids)

    def _bonus_scheme_ids(
        self,
        scheme_requests_collector: SchemeRequestsCollector,
        occupation_groups: Dict[OccupationId, OccupationDetails or None],
    ) -> Dict[ChangeSchemeRequest, ReviewSchemeId]:
        schemes_by_occupation = self._request_bonus_scheme_ids_by_occupation(scheme_requests_collector)
        schemes_by_occupation_groups = self._request_bonus_scheme_ids_by_occupation_groups(
            scheme_requests_collector,
            occupation_groups,
        )

        result = {}
        result.update(schemes_by_occupation)
        result.update(schemes_by_occupation_groups)
        return result

    def _review_scheme_ids(
        self,
        scheme_requests_collector: SchemeRequestsCollector,
        occupation_groups: Dict[OccupationId, OccupationDetails or None],
    ) -> Dict[ChangeSchemeRequest, ReviewSchemeId]:
        schemes_by_occupation = self._request_review_scheme_ids_by_occupation(scheme_requests_collector)
        schemes_by_occupation_groups = self._request_review_scheme_ids_by_occupation_groups(
            scheme_requests_collector,
            occupation_groups,
        )
        result = {}
        result.update(schemes_by_occupation_groups)
        result.update(schemes_by_occupation)
        return result

    def _request_review_scheme_ids_by_occupation_groups(
        self,
        scheme_requests_collector: SchemeRequestsCollector,
        occupation_groups: Dict[OccupationId, OccupationDetails or None],
    ) -> Dict[ChangeSchemeRequest, ReviewSchemeId]:
        requests = scheme_requests_collector.review_scheme_id_requests_by_occupation_groups(occupation_groups)
        if not requests:
            logger.info('No requests will be made for review scheme id by occupation groups')
            return {}

        review_scheme_ids = self._table_flow.review_scheme_id_by_group([request for _, request in requests])
        result = {}

        for (change_scheme_request, table_flow_request), review_scheme_id in zip(requests, review_scheme_ids):
            if review_scheme_id is not None:
                logger.info(
                    'Found review scheme id %s for change scheme request for person %s by group %s',
                    review_scheme_id,
                    change_scheme_request.person_id,
                    table_flow_request.occupation_review_group,
                )
                result[change_scheme_request] = review_scheme_id

        return result

    def _request_review_scheme_ids_by_occupation(
        self,
        scheme_requests_collector: SchemeRequestsCollector,
    ) -> Dict[ChangeSchemeRequest, ReviewSchemeId]:
        requests = scheme_requests_collector.review_scheme_id_requests_by_occupation()
        if not requests:
            logger.info('No requests will be made for review scheme id by occupation')
            return {}

        review_scheme_ids = self._table_flow.review_scheme_id([request for _, request in requests])
        result = {}

        # ответы приходят точно в таком же порядке, так мы узнаем к какому запросу относится схема
        for (change_scheme_request, request), review_scheme_id in zip(requests, review_scheme_ids):
            if review_scheme_id is not None:
                logger.info(
                    'Found review scheme id %s for requested change for person %s, occupation %s and department %s',
                    review_scheme_id,
                    change_scheme_request.person_id,
                    request.occupation_id,
                    request.department_id,
                )
                result[change_scheme_request] = review_scheme_id
            else:
                scheme_requests_collector.review_scheme_id_not_found_by_occupation(change_scheme_request)

        return result

    def _request_bonus_scheme_ids_by_occupation_groups(
        self,
        scheme_requests_collector: SchemeRequestsCollector,
        occupation_groups: Dict[OccupationId, OccupationDetails or None],
    ) -> Dict[ChangeSchemeRequest, BonusSchemeId]:
        table_flow_requests = scheme_requests_collector.bonus_scheme_id_requests_by_occupation_groups(occupation_groups)
        if not table_flow_requests:
            logger.info('No requests will be made for bonus scheme id by occupation groups')
            return {}

        bonus_scheme_ids = self._table_flow.bonus_scheme_id_by_group([request for _, request in table_flow_requests])
        result = {}

        for (change_scheme_request, table_flow_request), bonus_scheme_id in zip(table_flow_requests, bonus_scheme_ids):
            if bonus_scheme_id:
                logger.info(
                    'Found bonus scheme id %s for change scheme request for person %s by group %s',
                    bonus_scheme_id,
                    change_scheme_request.person_id,
                    table_flow_request.occupation_bonus_group,
                )
                result[change_scheme_request] = bonus_scheme_id
        return result

    def _request_bonus_scheme_ids_by_occupation(
        self,
        scheme_requests_collector: SchemeRequestsCollector,
    ) -> Dict[ChangeSchemeRequest, BonusSchemeId]:
        requests = scheme_requests_collector.bonus_scheme_id_requests_by_occupation()
        if not requests:
            logger.info('No requests will be made for bonus scheme id by occupation')
            return {}

        bonus_scheme_ids = self._table_flow.bonus_scheme_id([request for _, request in requests])
        result = {}

        for (change_scheme_request, table_flow_request), bonus_scheme_id in zip(requests, bonus_scheme_ids):
            if bonus_scheme_id is not None:
                logger.info(
                    'Found bonus scheme id %s for scheme change request for person %s, occupation %s and department %s',
                    bonus_scheme_id,
                    change_scheme_request.person_id,
                    change_scheme_request.occupation_id,
                    change_scheme_request.department_id,
                )
                result[change_scheme_request] = bonus_scheme_id
            else:
                logger.info(
                    'Not found bonus scheme for scheme change request for person %s, occupation %s and department %s',
                    change_scheme_request.person_id,
                    change_scheme_request.occupation_id,
                    change_scheme_request.department_id,
                )
                scheme_requests_collector.bonus_scheme_id_not_found_by_occupation(change_scheme_request)

        return result

    def _request_reward_categories_by_occupation_groups(
        self,
        scheme_requests_collector: SchemeRequestsCollector,
        occupation_groups: Dict[OccupationId, OccupationDetails or None],
    ) -> Dict[ChangeSchemeRequest, str]:
        requests = scheme_requests_collector.reward_category_requests_by_occupation_group(occupation_groups)
        if not requests:
            logger.info('No requests will be made for reward category by occupation group')
            return {}

        reward_categories = self._table_flow.reward_category([request for _, request in requests])
        result = {}

        for (change_scheme_request, reward_category_request), reward_category in zip(requests, reward_categories):
            if reward_category:
                logger.info(
                    'Found reward category %s for %s reward group',
                    reward_category,
                    reward_category_request.occupation_reward_group,
                )
                result[change_scheme_request] = reward_category
            else:
                logger.warning(
                    'Reward category not found for %s reward group',
                    reward_category_request.occupation_reward_group,
                )

        return result

    def _request_reward_scheme_ids_by_categories(
        self,
        scheme_requests_collector: SchemeRequestsCollector,
        categories: Dict[ChangeSchemeRequest, str],
    ) -> Dict[ChangeSchemeRequest, RewardSchemeId]:
        requests = scheme_requests_collector.reward_scheme_id_requests_by_category(categories)
        if not requests:
            logger.info('No requests will be made for reward scheme id by category')
            return {}

        result = {}
        reward_scheme_ids = self._table_flow.reward_scheme_id([request for _, request in requests])
        for (change_scheme_request, table_flow_request), reward_scheme_id in zip(requests, reward_scheme_ids):
            if reward_scheme_id:
                logger.info(
                    'Found reward scheme id %s for scheme change request for person %s',
                    reward_scheme_id,
                    change_scheme_request.person_id,
                )
                result[change_scheme_request] = reward_scheme_id
            else:
                logger.warning(
                    'Not found reward scheme id for scheme change request for person %s',
                    change_scheme_request.person_id,
                )

        return result

    def review_scheme_id(
        self,
        department_id: int,
        occupation_id: OccupationId,
        review_group: str,
        grade_level: int,
    ) -> Optional[ReviewSchemeId]:
        review_scheme_id = self._table_flow.review_scheme_id([ReviewSchemeIdRequest(
            occupation_id=occupation_id,
            department_id=department_id,
            grade_level=grade_level,
        )])[0]

        if review_scheme_id:
            logger.info(
                'Found review scheme id without using scheme group for occupation %s',
                occupation_id,
            )
            return review_scheme_id

        logger.info(
            'Scheme if for %s occupation not found, will try to find using review scheme group',
            occupation_id,
        )

        if review_group:
            review_scheme_id = self._table_flow.review_scheme_id_by_group([
                ReviewSchemeIdByGroupRequest(
                    occupation_review_group=review_group,
                    department_id=department_id,
                    grade_level=grade_level,
                )
            ])

            return review_scheme_id[0]

        logger.info('No review group for %s occupation and review scheme not found', occupation_id)
        return None

    def reward_scheme_id(
        self,
        reward_group: str,
        grade_level: int,
        department_id: int,
        is_internship: bool,
    ) -> Optional[RewardSchemeId]:
        category = self._table_flow.reward_category([RewardCategoryRequest(
            occupation_reward_group=reward_group,
            grade_level=grade_level,
        )])[0]

        if not category:
            logger.info('Cant find reward category %s for reward group', reward_group)
            return None

        logger.info('Found category %s for reward group %s in grade %s', category, reward_group, grade_level)

        reward_scheme_id = self._table_flow.reward_scheme_id([RewardSchemeIdRequest(
            department_id=department_id,
            category=category,
            is_internship=is_internship,
        )])

        return reward_scheme_id[0]

    def bonus_scheme_id(
        self,
        bonus_group: str,
        grade_level: int,
        department_id: int,
        occupation_id: OccupationId,
    ) -> Optional[BonusSchemeId]:
        bonus_scheme_id = self._table_flow.bonus_scheme_id([BonusSchemeIdRequest(
            department_id=department_id,
            occupation_id=occupation_id,
            grade_level=grade_level,
        )])[0]

        if bonus_scheme_id:
            logger.info(
                'Will use bonus scheme id found for occupation %s',
                occupation_id,
            )
            return bonus_scheme_id

        logger.info(
            'Bonus scheme if for %s occupation not found, will try to find using bonus scheme group',
            occupation_id,
        )

        if bonus_group:
            bonus_scheme_id = self._table_flow.bonus_scheme_id_by_group([BonusSchemeIdByGroupRequest(
                occupation_bonus_group=bonus_group,
                department_id=department_id,
                grade_level=grade_level,
            )])
            return bonus_scheme_id[0]

        logger.info('No bonus group for %s occupation and bonus scheme not found', occupation_id)
        return None

    def _has_enough_data_to_calculate_schemes(self, change: Change, data: FemidaData):
        if data.occupation_id is None:
            logger.info('Missing occupation while trying to calculate schemes')
            return False

        if data.grade_level is None:
            logger.info('Missing grade level while trying to calculate schemes')
            return False

        if change.department_id is None:
            logger.info('Missing department while trying to calculate schemes')
            return False

        return True

    def _is_new_vacancy(self, workflow: AbstractWorkflow):
        return workflow.code in (Workflow1_1.code, Workflow1_2.code, Workflow2_1.code)

    def _is_new_offer(self, workflow: AbstractWorkflow):
        return workflow.code in (Workflow5_1.code, Workflow5_3)

    def _is_proposal_workflow(self, workflow: AbstractWorkflow):
        return workflow.code == Workflow7_1.code
