import logging
from typing import Dict, List, Set

from staff.departments.models import Vacancy
from staff.person.models import Staff

from staff.budget_position.models import BudgetPosition, BudgetPositionAssignment
from staff.budget_position.workflow_service import entities


logger = logging.getLogger(__name__)


class BudgetPositionsRepository(entities.BudgetPositionsRepository):

    def get_or_create_new(self, code: entities.BudgetPositionCode) -> entities.BudgetPosition:
        result, _ = BudgetPosition.objects.get_or_create(code=code)
        return entities.BudgetPosition(id=result.id, code=result.code)

    def budget_position_by_code(self, code: entities.BudgetPositionCode) -> entities.BudgetPosition:
        model = BudgetPosition.objects.get(code=code)
        return entities.BudgetPosition(id=model.id, code=model.code)

    def budget_position_by_code_or_none(self, code: entities.BudgetPositionCode) -> entities.BudgetPosition or None:
        model = BudgetPosition.objects.filter(code=code).first()
        return entities.BudgetPosition(id=model.id, code=model.code) if model else None

    def budget_position_by_id(self, bp_id: int) -> entities.BudgetPosition or None:
        result = BudgetPosition.objects.filter(id=bp_id).first()
        return entities.BudgetPosition(id=result.id, code=result.code) if result else None

    def budget_positions_by_codes(self, codes: List[entities.BudgetPositionCode]) -> List[entities.BudgetPosition]:
        models = BudgetPosition.objects.filter(code__in=codes)
        return [
            entities.BudgetPosition(id=model.id, code=model.code)
            for model in models
        ]

    def assignments_statuses_by_codes(self, codes: List[entities.BudgetPositionCode]) -> Set[str]:
        return set(
            BudgetPositionAssignment.objects
            .active()
            .filter(budget_position__code__in=codes)
            .values_list('status', flat=True)
        )

    def headcount_position_data_by_vacancy_ids(
        self,
        vacancy_ids: List[int],
    ) -> Dict[int, entities.BudgetPosition]:
        bp_codes = list(
            Vacancy.objects
            .filter(pk__in=vacancy_ids)
            .values_list('budget_position__code', 'pk', 'budget_position_id')
        )

        if len(bp_codes) < len(vacancy_ids):
            found_bp_codes_for_vacancies = {vacancy_id for _, vacancy_id, _ in bp_codes}
            not_found_ids = set(vacancy_ids) - set(found_bp_codes_for_vacancies)
            logging.error('some vacancies not found %s', not_found_ids)
            raise entities.InvalidVacanciesIds(not_found_ids)

        result = {
            vacancy_id: entities.BudgetPosition(id=bp_id, code=bp_code)
            for bp_code, vacancy_id, bp_id in bp_codes
        }

        return result

    def headcount_position_data_by_persons_ids(
        self,
        person_ids: List[int],
    ) -> Dict[int, entities.BudgetPosition or None]:
        bp_codes_queryset = (
            Staff.objects
            .filter(pk__in=person_ids)
            .values_list('pk', 'budget_position__code', 'budget_position_id')
        )

        result = {}

        for person_id, code, bp_id in bp_codes_queryset:
            if bp_id is None:
                result[person_id] = None
            else:
                result[person_id] = entities.BudgetPosition(id=bp_id, code=code)

        return result
