import logging
from typing import Dict, List
from uuid import UUID

from django.conf import settings

from staff.departments.models import Vacancy
from staff.femida.constants import VACANCY_STATUS
from staff.femida.utils import get_vacancy_issue_key_from_url
from staff.lib.db import atomic
from staff.lib.log import log_context
from staff.lib.requests import Session

from staff.budget_position.models import FemidaPushOutbox
from staff.budget_position.models.change_registry import PUSH_STATUS
from staff.budget_position.workflow_service import entities, use_cases
from staff.budget_position.workflow_service.gateways.errors import FemidaError
from staff.budget_position.workflow_service.gateways.hr_product_resolver import HRProductResolver
from staff.budget_position.workflow_service.gateways.geography_resolver import GeographyResolver


logger = logging.getLogger(__name__)


class FemidaService(entities.FemidaService):
    def __init__(
        self,
        repository: entities.WorkflowRepositoryInterface,
        hr_product_resolver: HRProductResolver = None,
        geography_resolver: GeographyResolver = None,
        startrek_service: use_cases.StartrekService = None,
    ) -> None:
        self._repository = repository
        self._hr_product_resolver = hr_product_resolver or HRProductResolver()
        self._geography_resolver = geography_resolver or GeographyResolver()

        from staff.budget_position.workflow_service.gateways import StartrekService
        self._startrek_service = startrek_service or StartrekService()

        self._session = Session()
        self._session.headers.update({
            'Authorization': 'OAuth {token}'.format(token=settings.ROBOT_STAFF_OAUTH_TOKEN),
        })

    def schedule_department_push(self, workflow_id: UUID):
        FemidaPushOutbox.objects.create(workflow_id=workflow_id)
        logger.info('Workflow %s scheduled for pushing to femida', workflow_id)

    def change_vacancy_url(self, vacancy_id: int):
        return f'https://{settings.FEMIDA_HOST}/_api/staff/vacancies/{vacancy_id}/'

    def change_bp_url(self, vacancy_id: int):
        return f'https://{settings.FEMIDA_HOST}/_api/staff/vacancies/{vacancy_id}/create_bp/'

    def close_vacancy_url(self, vacancy_id: int):
        return f'https://{settings.FEMIDA_HOST}/_api/staff/vacancies/{vacancy_id}/close/'

    def close_vacancy(self, workflow_id: UUID):
        with log_context(workflow_id=str(workflow_id)):
            workflow = self._repository.get_by_id(workflow_id)
            if not workflow.vacancy_id or not workflow.vacancy.is_active:
                logger.error('Workflow has no active vacancy')
                return
            if not workflow.credit_management_id:
                logger.error('Trying to close vacancy for workflow without credit management')
                return

            author = workflow.credit_management.author.login
            ticket = workflow.credit_management.startrek_headcount_key
            comment = f'Согласовано закрытие кредита в тикете {ticket} (автор staff:{author})'

            result = self._session.post(
                self.close_vacancy_url(workflow.vacancy_id),
                json={'comment': comment},
                timeout=4,
                no_log_json=True,
            )

            if result.status_code != 200:
                raise FemidaError("Не удалось закрыть вакансию в фемиде")

            workflow.vacancy.status = VACANCY_STATUS.CLOSED
            workflow.vacancy.is_active = False
            workflow.vacancy.save()

    def _log_result_if_needed(self, result, workflow: entities.AbstractWorkflow):
        if result.status_code == 304:
            logger.info(
                'For workflow %s Femida did not modify %s vacancy',
                workflow.id,
                workflow.vacancy_id,
            )
            return

        if result.status_code != 200:
            logger.error(
                'For workflow %s Femida returned %s: %s',
                workflow.id,
                result.status_code,
                result.content and result.content.decode(),
            )

    def _should_retry_later(self, result) -> bool:
        if result.status_code == 304:
            return False

        if result.status_code != 200:
            return True

        return False

    @staticmethod
    def _can_push_new_bp_to_femida(workflow: entities.AbstractWorkflow) -> bool:
        return workflow.should_push_new_bp_to_femida and workflow.changes[1].push_status == PUSH_STATUS.FINISHED

    def _push_vacancy_update_to_femida(self, workflow: entities.AbstractWorkflow) -> bool:
        change_id = workflow.changes[0]
        department_id = change_id.department_id
        hr_product_id = change_id.hr_product_id
        value_stream_id, value_stream_error = self._hr_product_resolver.resolve_hr_product(hr_product_id)
        geography_id, geography_error = self._geography_resolver.resolve_geography(change_id.geography_url)
        ticket = change_id.ticket

        if department_id is None and value_stream_id is None and geography_id is None:
            logger.info('No data to sent to Femida in change %s', change_id.id)
            return True

        data = {
            'department': department_id,
            'value_stream': value_stream_id,
            'geography': geography_id,
            'startrek_approval_issue_key': ticket,
        }

        post_result = self._session.post(
            self.change_vacancy_url(workflow.vacancy_id),
            json=data,
            timeout=4,
            no_log_json=True,
        )

        self._log_result_if_needed(post_result, workflow)
        result = not self._should_retry_later(post_result)

        if not (value_stream_error or geography_error):
            return result

        comments = []
        if value_stream_error:
            logger.info('VS resolving error %s, ticket %s', value_stream_error, ticket)
            comments += [f'Основной продукт (Value Stream) не может быть обновлен в Фемиде\n{value_stream_error}']

        if geography_error:
            logger.info('Geography resolving error %s, ticket %s', geography_error, ticket)
            comments += [f'География задач (Geography) не может быть обновлен в Фемиде\n{geography_error}']

        if ticket:
            self._startrek_service.add_comment(ticket, '\n\n'.join(comments))
        return result

    def _push_new_bp_to_femida(self, workflow: entities.AbstractWorkflow) -> bool:
        budget_position = workflow.changes[1].new_budget_position.code
        result = self._session.post(
            self.change_bp_url(workflow.vacancy_id),
            json={'budget_position_id': budget_position},
            timeout=4,
            no_log_json=True,
        )

        self._log_result_if_needed(result, workflow)
        return not self._should_retry_later(result)

    @atomic
    def _push_one_workflow(self, scheduled_push: FemidaPushOutbox):
        workflow_id = scheduled_push.workflow.id
        with log_context(workflow_id=str(workflow_id)):
            try:
                workflow = self._repository.get_by_id(workflow_id)

                if not workflow.vacancy_id:
                    logger.error('Workflow has no active vacancy')
                    # мониторинг на переполнение PushOutbox?
                    return

                handling_finished = True
                if workflow.should_push_new_department_to_femida:
                    handling_finished &= self._push_vacancy_update_to_femida(workflow)

                if workflow.should_push_new_bp_to_femida:
                    if self._can_push_new_bp_to_femida(workflow):
                        handling_finished &= self._push_new_bp_to_femida(workflow)
                    else:
                        logger.info('Change not finished yet to push new BP to Femida')
                        handling_finished = False

                if handling_finished:
                    logger.info('FemidaPushOutbox handling finished')
                    scheduled_push.delete()

            except Exception:
                logger.exception('Unhandled exception on push to Femida')
                raise

    def push_scheduled(self):
        for scheduled in FemidaPushOutbox.objects.all():
            self._push_one_workflow(scheduled)

    def vacancies_by_job_ticket_urls(self, job_ticket_urls: List[str]) -> Dict[str, int]:
        issue_keys = {
            get_vacancy_issue_key_from_url(job_ticket_url): job_ticket_url
            for job_ticket_url in job_ticket_urls
        }

        vacancies = (
            Vacancy.objects
            .filter(ticket__in=[issue_key for issue_key in issue_keys.keys() if issue_key])
            .values_list('pk', 'ticket')
        )

        result = {}
        for vacancy_id, vacancy_issue in vacancies:
            if vacancy_issue in issue_keys:
                job_ticket_url = issue_keys[vacancy_issue]
                result[job_ticket_url] = vacancy_id

        if len(result) < len(job_ticket_urls):
            logger.warning('Not all vacancies found by tickets, found %s', list(result.keys()))

        return result
