from datetime import date, datetime
from decimal import Decimal
import json
import logging
from typing import Any, Dict, List, Optional, Tuple

from staff.lib import waffle
import yenv
from dateutil import parser
from django.conf import settings

from staff.departments.models import Geography
from staff.lib import requests
from staff.lib.tvm2 import TVM_SERVICE_TICKET_HEADER, get_tvm_ticket_by_deploy

from staff.person.models import Organization

from staff.budget_position.const import PUSH_STATUS, WORKFLOW_STATUS, FemidaProfessionalLevel
from staff.budget_position.models import OEBSGrade, ProfLevelMapping, GeographyMapping
from staff.budget_position.workflow_service import entities, errors


logger = logging.getLogger(__name__)


class OEBSService(entities.OEBSService):
    """
    Stateless domain service, contains logic for pushing workflows to OEBS.
    This logic does not naturally fit elsewhere
    """

    def __init__(self, budget_positions_repository: entities.interfaces.BudgetPositionsRepository):
        self._bp_repository = budget_positions_repository

    def push_next_change_to_oebs(self, workflow: entities.AbstractWorkflow, catalyst_login: str) -> None:
        if workflow.status not in (WORKFLOW_STATUS.PUSHED, WORKFLOW_STATUS.SENDING_NOTIFICATION):
            logger.warning(f'Workflow {workflow.id} has unexpected status {workflow.status}.')
            raise entities.WorkflowInvalidStateError(workflow.id, workflow.status)

        change_to_send = workflow.get_next_change_for_sending()

        if change_to_send:
            self.send_change_to_oebs(change_to_send, workflow, catalyst_login)
        elif workflow.is_finished():
            workflow.mark_pushed_to_oebs()
        else:
            logger.info(f"Waiting workflow {workflow.id} changes to be applied.")

    def update_changes_push_status(self, changes: List[entities.Change]) -> bool:
        """ returns `True` if no pushing changes"""

        has_no_pushing_changes = True

        for change in changes:
            if change.push_status == PUSH_STATUS.PUSHED:
                assert change.oebs_transaction_id is not None
                status, budget_position, correction_id, error = self.get_transaction_status(change.oebs_transaction_id)
                change.last_oebs_error = error
                push_status = self._get_push_status_by_oebs_transaction_status(status)

                if push_status:
                    change.push_status = push_status

                    if change.budget_position is None and push_status == PUSH_STATUS.FINISHED:
                        change.new_budget_position = self._bp_repository.get_or_create_new(budget_position)

                    if correction_id:
                        change.correction_id = correction_id
                else:
                    has_no_pushing_changes = False

        return has_no_pushing_changes

    @staticmethod
    def _get_push_status_by_oebs_transaction_status(status: str) -> Optional[str]:
        return {
            'UPLOADED': PUSH_STATUS.FINISHED,
            'ERROR': PUSH_STATUS.ERROR,
        }.get(status)

    @staticmethod
    def _patched_ticket(ticket: Optional[str], sequence: int) -> str:
        if ticket is None:
            return f'TSALARY-100000000000000{sequence}'

        if yenv.type == 'production':
            return ticket

        if ticket.startswith('TSALARY') or ticket.startswith('TJOB'):
            return ticket[1:] + '0000'

        return ticket

    def _oebs_geography_code(self, geography_url: Optional[str]) -> Optional[str]:
        if not geography_url:
            return None

        mapping = GeographyMapping.objects.filter(staff_geography__department_instance__url=geography_url).first()
        if mapping:
            return mapping.oebs_geography_code
        geography = Geography.objects.filter(department_instance__url=geography_url).first()
        return geography and geography.oebs_code

    def _geography_url(self, geography_oebs_code: str) -> Optional[str]:
        if not geography_oebs_code:
            return None

        result = Geography.objects.filter(oebs_code=geography_oebs_code).first()

        if not result:
            return None

        return result.department_instance.url

    def _change_can_be_sent_to_oebs(self, change: entities.Change) -> bool:
        return any((
            change.creates_new_budget_positions,
            change.changes_existing_budget_position,
            change.moves_from_one_budget_position_to_another,
        ))

    def _budget_position_code_to_send(self, change: entities.Change) -> Optional[entities.BudgetPositionCode]:
        if change.moves_from_one_budget_position_to_another:
            return change.new_budget_position.code

        if change.creates_new_budget_positions:
            return None

        return change.budget_position.code

    def _make_oebs_data_for_change(
        self,
        change: entities.Change,
        workflow: entities.AbstractWorkflow,
        catalyst_login: str,
    ) -> Dict[str, Any]:
        if workflow.catalyst_id is None or not catalyst_login:
            logger.info('No catalyst defined')
            raise errors.NoHRAnalyst('No catalyst defined')

        assert self._change_can_be_sent_to_oebs(change)
        assert workflow.permission_date is not None
        assert change.id is not None

        organization = (
            Organization.objects
            .filter(id=change.organization_id)
            .select_related('oebs_organization')
            .first()
        )
        # Связанная организация обязана быть, если ее нет, то с синком что-то не так
        oebs_organization = organization and organization.oebs_organization

        linked_budget_position = self._bp_repository.budget_position_by_id(change.linked_budget_position_id)

        salary = change.salary
        if salary is not None:
            salary = float(salary)

        rate = change.rate
        if rate is not None:
            rate = float(round(rate, 3))

        data = {
            'apiUniqueKey': str(change.oebs_idempotence_key),
            'login': catalyst_login,
            'loginDate': workflow.permission_date.isoformat(),
            'positionNumber': self._budget_position_code_to_send(change),
            'positionName': change.position_name,
            'ticket': self._patched_ticket(change.ticket, change.id),
            'forRecruitment': {
                'dessmissalDate': change.dismissal_date and change.dismissal_date.isoformat(),
                'hireDate': date.today().isoformat(),
                'positionPermission': change.position_type and change.position_type.name,
            },
            'positionAnalytics': [{
                'unitManager': {True: 'Y', False: 'N', None: None}.get(change.unit_manager),
                'headCount': change.headcount,
                'effectiveDate': change.effective_date and str(change.effective_date),
                'statusID': 5 if change.remove_budget_position else None,
                'currency': change.currency,
                'geography': self._oebs_geography_code(change.geography_url),
                'gradeID': change.grade_id,
                'organizationID': change.department_id,
                'paySystem': change.pay_system,
                'salary': salary,
                'productID': change.hr_product_id,
                'rwsBonusID': change.bonus_scheme_id,
                'rwsRewardID': change.reward_scheme_id,
                'rwsReviewID': change.review_scheme_id,
                'fte': rate,
                'taxUnitID': change.placement_id and oebs_organization and oebs_organization.org_id,
                'officeID': change.placement_id,
                'linkedPos': linked_budget_position and linked_budget_position.code,
            }],
        }

        dismissal_date_to_send = data['forRecruitment']['dessmissalDate']
        effective_date = data['positionAnalytics'][0]['effectiveDate']
        if dismissal_date_to_send and effective_date and dismissal_date_to_send < effective_date:
            data['forRecruitment']['dessmissalDate'] = None

        def clean_data(*dicts: dict):
            for dict_ in dicts:
                for key, value in dict_.copy().items():
                    if value is None:
                        dict_.pop(key)

        clean_data(data['forRecruitment'], data['positionAnalytics'][0], data)

        # В след. релизе OEBS будет исправлена орфография
        # Пока передаем оба ключа, чтобы не синхронизировать выкатки
        data['positionAnalitics'] = data['positionAnalytics']

        return data

    def send_change_to_oebs(
        self,
        change: entities.Change,
        workflow: entities.AbstractWorkflow,
        catalyst_login: str,
    ) -> None:
        data = self._make_oebs_data_for_change(change, workflow, catalyst_login)

        try:
            result = self.send_request(
                settings.OEBS_PUSH_POSITION_URL,
                data,
                f'Sending bp change {change.id} to oebs',
            )
            transaction_id = result.get('ID')

            if transaction_id:
                change.oebs_transaction_id = transaction_id
                change.last_oebs_error = None
                change.sent_to_oebs = datetime.now()
                change.push_status = PUSH_STATUS.PUSHED
                return

            last_oebs_error = result.get('error')
            change.last_oebs_error = last_oebs_error
            raise entities.OebsErrorResponseOEBSError(last_oebs_error)
        except ValueError as e:
            last_error = entities.UnexpectedResponseOEBSError(str(e))
            logger.info(
                'Unexpected OEBS response when sending bp change (%s) to oebs raised error %s',
                change.id,
                last_error,
            )
            raise last_error
        except entities.NoResponseOEBSError as e:
            last_error = e
            logger.info('Error when sending bp change (%s) to oebs: %s', change.id, last_error)
            raise last_error

    @staticmethod
    def send_request(url: str, data: Any, log_message: Optional[str] = None, timeout=10) -> Any:
        try:
            response = requests.post(
                url=url,
                headers={
                    TVM_SERVICE_TICKET_HEADER: get_tvm_ticket_by_deploy('oebs-api'),
                    'Content-Type': 'application/json',
                },
                log_message=log_message,
                timeout=timeout,
                data=json.dumps(data),
            )
            response.raise_for_status()
            return response.json()
        except (requests.HTTPError, requests.Timeout) as e:
            raise entities.NoResponseOEBSError(str(e))

    def get_position_link(self, change: entities.Change, login: str, is_write=False) -> str:
        assert change

        new_budget_position_code = change.new_budget_position and change.new_budget_position.code
        budget_position_code = change.budget_position and change.budget_position.code
        result = new_budget_position_code or budget_position_code
        assert result

        request = {
            'login': login,
            'method': 'WRITE' if is_write else 'READ',
            'bpNumber': result,
            'effectiveDate': change.effective_date and str(change.effective_date),
        }

        result = self._oebs_post_request(settings.OEBS_GET_POSITION_LINK_URL, request)
        return self._get_link(result)

    def get_budget_link(self, change: entities.Change) -> str:
        if not change.correction_id:
            raise entities.OEBSError('В изменении нет номера корректировки')

        result = self._oebs_post_request(settings.OEBS_GET_BUDGET_LINK_URL, {'CorrectionID': change.correction_id})
        return self._get_link(result)

    def _get_link(self, response) -> str:
        link = response.get('link')

        if not link:
            raise entities.UnexpectedResponseOEBSError('OEBS не вернул ссылку')

        return link

    def get_transaction_status(self, oebs_transaction_id: int) -> Tuple[str, int, int, str]:
        try:
            response = requests.post(
                url=settings.OEBS_GET_TRANSACTION_STATUS_URL,
                headers={TVM_SERVICE_TICKET_HEADER: get_tvm_ticket_by_deploy('oebs-api')},
                timeout=(1, 3, 5),
                json={'BatchID': oebs_transaction_id},
            )
            result = response.json()
        except (ValueError, requests.HTTPError, requests.Timeout):
            raise entities.NoResponseOEBSError('OEBS не отвечает')

        status = result.get('ProcessingStatus')
        budget_position = result.get('PositionNumber')
        correction_id = result.get('CorrectionId')
        error = result.get('Error')
        logger.info(f'OEBS transaction {oebs_transaction_id} status is {status}: {error}')

        if not status:
            raise entities.UnexpectedResponseOEBSError('OEBS не вернул статус')

        return status, budget_position, correction_id, error

    def get_salary_data(self, login: str) -> entities.SalaryData:
        if waffle.switch_is_active('enable_oebs_mocked_data'):
            return entities.SalaryData(salary=Decimal(12345), rate=Decimal(1))

        result = self._oebs_post_request(settings.OEBS_GET_SALARY_URL, {'login': [login]})
        salaries = result.get('salaries', [])

        if not salaries:
            raise entities.UnexpectedResponseOEBSError('No salaries')

        salary = salaries[0]
        if salary['salarySum'] is None or salary['factFTE'] is None:
            raise entities.UnexpectedResponseOEBSError('No actual salary returned')

        data = entities.SalaryData(
            salary=salary['salarySum'],
            rate=salary['factFTE'],
            currency=salary['currency'],
        )

        if yenv.type != 'production' and not waffle.switch_is_active('enable_oebs_actual_salary_data'):
            data.salary = Decimal(12345)

        return data

    def get_grades_data(self, logins: List[str]) -> Dict[str, Optional[entities.GradeData]]:
        assert bool(logins) and 'non empty logins list required'
        if waffle.switch_is_active('enable_oebs_mocked_data'):
            return {
                login: entities.GradeData(occupation='SomeOccupation', level=10)
                for login in logins
            }

        try:
            result = self._oebs_post_request(settings.OEBS_GET_SALARY_URL, {'login': logins})
        except entities.EmptyResponseOEBSError:
            return {login: None for login in logins}

        raw_data = result.get('salaries', [])
        if not raw_data:
            logger.error('Error getting grade data for persons %s', logins)
            raise entities.UnexpectedResponseOEBSError('No salaries')

        grades_data = {}
        for data in raw_data:
            grades_data[data['login']] = self._parse_grade(data['gradeName'])

        if yenv.type != 'production' and not waffle.switch_is_active('enable_oebs_actual_salary_data'):
            for data in grades_data.values():
                data.level = None

        return grades_data

    def _parse_grade(self, grade_name: str) -> Optional[entities.GradeData]:
        # format - 'CallCentreSpec.9.2'
        # 'Без грейда' представлен как None
        if not grade_name or not isinstance(grade_name, str):
            return None

        parts = grade_name.split('.')
        if len(parts) == 3:
            occupation, level, _ = parts
        elif len(parts) == 2:
            occupation, level = parts
        else:
            occupation, level = None, None

        if not occupation:
            return None

        level = int(level) if level.isdigit() else None
        return entities.GradeData(occupation=occupation, level=level)

    def get_grade_id(self, occupation: entities.OccupationId, level: Optional[int]) -> Optional[int]:
        return (
            OEBSGrade.objects
            .filter(occupation=occupation, level=level)
            .values_list('grade_id', flat=True)
            .first()
        )

    def get_grade_data(self, grade_id: entities.GradeId) -> Optional[entities.GradeData]:
        oebs_grade = OEBSGrade.objects.filter(grade_id=grade_id).first()
        if oebs_grade:
            return entities.GradeData(occupation=oebs_grade.occupation_id, level=oebs_grade.level)

        return None

    def get_grade_id_by_femida_level(
        self,
        occupation: entities.OccupationId,
        level: FemidaProfessionalLevel,
    ) -> Optional[int]:
        mapping = (
            ProfLevelMapping.objects
            .filter(occupation=occupation)
            .first()
        )

        if not mapping:
            logger.warning('Level mapping for occupation %s not found', occupation)
            return None

        result = getattr(mapping, level.name, None)
        if not result:
            logger.warning('Level mapping for occupation %s and level %s not found', occupation, level)
        return result

    def get_position_as_change(self, budget_position_code: entities.BudgetPositionCode) -> entities.Change:
        request = {'positionNumber': budget_position_code, 'effectiveDate': datetime.today().isoformat()}
        result = self._oebs_post_request(settings.OEBS_GET_POSITION_URL, request)

        recruitment = result['forRecruitment']
        analytics = result['positionAnalitics']

        result = entities.Change(
            budget_position=result.get('positionNumber'),
            position_name=result.get('positionName'),
            ticket=result.get('ticket'),
            dismissal_date=recruitment.get('dessmissalDate') and parser.parse(recruitment['dessmissalDate']).date(),
            position_type=recruitment.get('positionPermission') and recruitment['positionPermission'].lower(),
            unit_manager={'Y': True, 'N': False}.get(analytics.get('unitManager')),
            headcount=analytics.get('headCount'),
            currency=analytics.get('currency'),
            geography_url=self._geography_url(analytics.get('geography')),
            grade_id=analytics.get('gradeID'),
            # department_id=analytics.get('organizationID'), TODO: тут id не со стаффа -_-
            pay_system=analytics.get('paySystem'),
            salary=analytics.get('salary') and str(analytics['salary']),
            hr_product_id=analytics.get('productID'),
            bonus_scheme_id=analytics.get('rwsBonusID'),
            reward_scheme_id=analytics.get('rwsRewardID'),
            review_scheme_id=analytics.get('rwsReviewID'),
            rate=analytics.get('fte') and str(analytics['fte']),
            placement_id=analytics.get('officeID'),
        )

        if analytics.get('linkedPos'):
            bp = self._bp_repository.budget_position_by_code(analytics.get('linkedPos'))
            result.linked_budget_position_id = bp.id

        if analytics.get('taxUnitID'):
            organization_id = (
                Organization.objects
                .filter(oebs_organization__org_id=analytics['taxUnitID'])
                .values_list('id', flat=True)
                .get()
            )
            result.organization_id = organization_id

        return result

    def get_crossing_position_info_as_change(self, budget_position_code: int, login: str) -> Optional[entities.Change]:
        request = {
            'effectiveDate': date.today().isoformat(),
            'positionNumber': budget_position_code,
            'login': login,
            'ovlType': 'A',
            'limit': 1,
            'offset': 0,
        }

        response = self._oebs_post_request(settings.OEBS_GET_POSITION_ADD_INFO_URL, request)
        data = response.get('addInfData')
        if not data:
            return None

        analytics = data[0].get('addInfAnalytics')
        if not analytics:
            return None

        analytics = analytics[0]

        result = entities.Change(
            grade_id=analytics.get('gradeID'),
            salary=analytics.get('salary') and str(analytics['salary']),
            currency=analytics.get('currency'),
            rate=analytics.get('fte') and str(analytics['fte']),
            department_id=analytics.get('organizationID'),
        )

        return result

    def _oebs_post_request(self, url: str, request: Dict) -> Dict:
        try:
            response = requests.post(
                url=url,
                headers={TVM_SERVICE_TICKET_HEADER: get_tvm_ticket_by_deploy('oebs-api')},
                timeout=(1, 3, 5),
                json=request,
            )

            if not response.content:
                raise entities.EmptyResponseOEBSError()

            result = response.json()

            error = result.get('error')

            if error:
                raise entities.OebsErrorResponseOEBSError('OEBS: {}'.format(error))

            if result.get('status') == 'ERROR':
                raise entities.OebsErrorResponseOEBSError('OEBS: {}'.format(result.get('message', '')))

            return result

        except requests.Timeout:
            raise entities.NoResponseOEBSError('OEBS: Timeout')

        except (ValueError, requests.HTTPError):
            raise entities.OEBSError()

        except json.JSONDecodeError:
            raise entities.WrongJsonResponseOEBSError()
