import logging
import uuid
from typing import List, Optional
from uuid import UUID

from django.db.models import Q
from ylog.context import log_context

from staff.departments.models import Geography
from staff.departments.models.headcount import HeadcountPosition
from staff.person.models import Staff

from staff.budget_position.const import PUSH_STATUS, WORKFLOW_STATUS, PositionType
from staff.budget_position.models import ChangeRegistry, Workflow
from staff.budget_position.workflow_service import entities
from staff.budget_position.workflow_service.entities import workflows


logger = logging.getLogger(__name__)


class WorkflowRepository(entities.WorkflowRepositoryInterface):
    """
    Repository is a domain-aware interface into an external data storage
    Responsible for CRUD operations.
    A repository is a persistent container of aggregates, thud repository always save and retrieve complete aggregates.
    """

    workflows_by_code = {
        workflows.Workflow1_1.code: workflows.Workflow1_1,
        workflows.Workflow1_2.code: workflows.Workflow1_2,
        workflows.Workflow1_3.code: workflows.Workflow1_3,
        workflows.Workflow2_1.code: workflows.Workflow2_1,
        workflows.Workflow5_1.code: workflows.Workflow5_1,
        workflows.Workflow5_1_1.code: workflows.Workflow5_1_1,
        workflows.Workflow5_2.code: workflows.Workflow5_2,
        workflows.Workflow5_3.code: workflows.Workflow5_3,
        workflows.Workflow7_1.code: workflows.Workflow7_1,
        workflows.Workflow1_1_1.code: workflows.Workflow1_1_1,
        workflows.MoveWithoutBudgetPositionWorkflow.code: workflows.MoveWithoutBudgetPositionWorkflow,
        workflows.Workflow7_100500.code: workflows.Workflow7_100500,
        workflows.WorkflowCreditManagementWithVacancy.code: workflows.WorkflowCreditManagementWithVacancy,
        workflows.WorkflowCreditManagement.code: workflows.WorkflowCreditManagement,
        workflows.HeadcountChangeFromProposalWorkflow.code: workflows.HeadcountChangeFromProposalWorkflow,
        workflows.MaternityWorkflow.code: workflows.MaternityWorkflow,
    }

    def save(self, workflow: entities.AbstractWorkflow) -> UUID:
        with log_context(workflow_id=str(workflow.id)):
            defaults = {
                'manually_processed': workflow.manually_processed,
                'code': workflow.code,
                'status': workflow.status,
                'proposal_id': workflow.proposal_id,
                'vacancy_id': workflow.vacancy_id,
                'credit_management_id': workflow.credit_management_id,
                'confirmed_at': workflow.confirmed_at,
                'catalyst_id': workflow.catalyst_id,
                'permission_date': workflow.permission_date,
            }

            workflow_model, created = Workflow.objects.update_or_create(id=workflow.id, defaults=defaults)
            logger.info('Workflow %s', 'saved' if created else 'updated')

            for change in workflow.changes:
                change.workflow_id = str(workflow.id)
                self._change_to_model(change).save()

            logger.info('Saved changes')

            return workflow_model.id

    def get_pending_workflows_by_budget_position_code(
        self,
        budget_position_code: int or None,
    ) -> List[entities.AbstractWorkflow]:
        if not budget_position_code:
            return []

        q = Q(changeregistry__budget_position__code=budget_position_code, status=WORKFLOW_STATUS.PENDING)

        workflow_models = Workflow.objects.prefetch_related('changeregistry_set').filter(q)
        return [self._create_workflow_from_db(model) for model in workflow_models]

    def get_by_id(self, workflow_id: UUID) -> entities.AbstractWorkflow:
        workflow_model = Workflow.objects.select_related('catalyst').get(id=workflow_id)
        return self._create_workflow_from_db(workflow_model)

    def get_workflows_by_proposal_id(
        self,
        proposal_id: int,
        status: str = WORKFLOW_STATUS.PENDING,
    ) -> List[entities.AbstractWorkflow]:
        q = Q(proposal_id=proposal_id)

        if status:
            q &= Q(status=status)

        workflow_queryset = Workflow.objects.prefetch_related('changeregistry_set').filter(q)
        return [self._create_workflow_from_db(workflow) for workflow in workflow_queryset]

    def get_workflows_by_credit_management_application(
        self,
        credit_repayment_application_id: int,
    ) -> List[entities.AbstractWorkflow]:
        q = Q(credit_management_id=credit_repayment_application_id)
        workflow_queryset = Workflow.objects.prefetch_related('changeregistry_set').filter(q)
        return [self._create_workflow_from_db(workflow) for workflow in workflow_queryset]

    def cancel_workflows_for_proposal(self, proposal_id: int) -> None:
        workflow_q = Q(
            proposal_id=proposal_id,
            status=WORKFLOW_STATUS.PENDING,
        )
        updated_workflows = Workflow.objects.filter(workflow_q).update(status=WORKFLOW_STATUS.CANCELLED)

        logger.info('Cancelled %s workflows for proposal_id %s', updated_workflows, proposal_id)

    def cancel_workflows_for_credit_repayment(self, credit_repayment_application_id: int) -> None:
        workflow_q = Q(
            credit_management_id=credit_repayment_application_id,
            status=WORKFLOW_STATUS.PENDING,
        )
        updated_workflows = Workflow.objects.filter(workflow_q).update(status=WORKFLOW_STATUS.CANCELLED)
        logger.info(
            'Cancelled %s workflows for credit repayment %s',
            updated_workflows,
            credit_repayment_application_id,
        )

    def queue_workflows(self, workflow_ids: List[UUID], person: Optional[Staff]) -> None:
        workflow_q = Q(id__in=workflow_ids) & Q(status=WORKFLOW_STATUS.CONFIRMED)
        target_workflows = Workflow.objects.filter(workflow_q)

        if person is not None:
            target_workflows.update(status=WORKFLOW_STATUS.QUEUED, catalyst=person)
        else:
            target_workflows.update(status=WORKFLOW_STATUS.QUEUED)

        logger.info('Queued workflows %s', workflow_ids)

    def retry_workflows(self, workflow_ids: List[UUID], person_id: int) -> None:
        workflow_q = Q(id__in=workflow_ids) & Q(status=WORKFLOW_STATUS.FINISHED)
        target_workflows = Workflow.objects.filter(workflow_q)
        target_changes = ChangeRegistry.objects.filter(workflow__in=target_workflows, push_status=PUSH_STATUS.ERROR)

        for target_change in target_changes:
            target_change.push_status = None
            target_change.oebs_transaction_id = None
            target_change.last_oebs_error = None
            target_change.oebs_idempotence_key = uuid.uuid1()
            target_change.save()

        target_workflows.update(status=WORKFLOW_STATUS.QUEUED, catalyst_id=person_id)

        logger.info('Re-queued workflows %s', workflow_ids)

    def get_workflow_list(self, workflow_ids: List[UUID]) -> List[entities.AbstractWorkflow]:
        target_workflows = Workflow.objects.filter(id__in=workflow_ids).select_related('catalyst')
        return [self._create_workflow_from_db(workflow) for workflow in target_workflows]

    def is_all_workflows_for_credit_repayment_not_in_pending(self, credit_management_id: int) -> bool:
        statuses = (WORKFLOW_STATUS.FINISHED, WORKFLOW_STATUS.SENDING_NOTIFICATION, WORKFLOW_STATUS.CANCELLED)
        has_not_pushed_workflows = (
            Workflow.objects
            .exclude(status__in=statuses)
            .filter(credit_management_id=credit_management_id)
            .exists()
        )
        return not has_not_pushed_workflows

    def is_all_workflows_for_credit_repayment_is_cancelled(self, credit_management_id: int) -> bool:
        not_cancelled_workflows_exists = (
            Workflow.objects
            .filter(credit_management_id=credit_management_id)
            .exclude(status=WORKFLOW_STATUS.CANCELLED)
            .exists()
        )
        return not not_cancelled_workflows_exists

    def get_workflows(self, status: str) -> List[UUID]:
        return list(Workflow.objects.filter(status=status).values_list('id', flat=True))

    def can_workflow_be_finalized(self, workflow_id: UUID) -> bool:
        return self.get_by_id(workflow_id).is_finished()

    def get_related_department_ids(self, workflow_id: UUID) -> List[int]:
        workflow = (
            Workflow.objects
            .get(id=workflow_id)
        )
        bp_codes = workflow.changeregistry_set.values_list('budget_position__code', flat=True)
        headcounts = HeadcountPosition.objects.filter(code__in=bp_codes)
        return list(headcounts.values_list('department_id', flat=True))

    def get_related_tickets(self, workflow_id: UUID) -> List[str]:
        workflow = (
            Workflow.objects
            .prefetch_related('changeregistry_set')
            .get(id=workflow_id)
        )
        return [change.ticket for change in workflow.changeregistry_set.all()]

    def mark_changes_as_failed(self, workflow_id: UUID, exc: Exception) -> None:
        workflow = Workflow.objects.get(id=workflow_id)
        updated = workflow.changeregistry_set.filter(push_status=None).update(
            push_status=PUSH_STATUS.ERROR,
            last_oebs_error=str(exc),
        )
        logger.info('Marked %s changes of workflow %s as failed', updated, str(workflow_id))

    def _geography_url(self, db_model: ChangeRegistry) -> str or None:
        return db_model.geo and db_model.geo.department_instance.url

    def _change_from_model(self, db_model: ChangeRegistry) -> entities.Change:
        budget_position = db_model.budget_position
        new_budget_position = db_model.new_budget_position
        geography_url = self._geography_url(db_model)

        res = entities.Change(
            id=db_model.id,
            budget_position=budget_position and entities.BudgetPosition(budget_position.id, budget_position.code),
            new_budget_position=(
                new_budget_position and entities.BudgetPosition(new_budget_position.id, new_budget_position.code)
            ),
            linked_budget_position_id=db_model.linked_budget_position_id,
            pushed_to_femida=db_model.pushed_to_femida,
            oebs_transaction_id=db_model.oebs_transaction_id,
            oebs_idempotence_key=db_model.oebs_idempotence_key,
            last_oebs_error=db_model.last_oebs_error,
            correction_id=db_model.correction_id,
            push_status=db_model.push_status,
            sent_to_oebs=db_model.sent_to_oebs,
            assignment_id=db_model.assignment_id,
            workflow_id=str(db_model.workflow_id),
            bonus_scheme_id=db_model.staff_bonus_scheme_id,
            reward_scheme_id=db_model.compensation_scheme_id,
            currency=db_model.currency,
            department_id=db_model.department_id,
            dismissal_date=db_model.dismissal_date,
            effective_date=db_model.effective_date,
            geography_url=geography_url,
            grade_id=db_model.oebs_grade_id,
            headcount=db_model.headcount,
            hr_product_id=db_model.staff_hr_product_id,
            office_id=db_model.office_id,
            organization_id=db_model.organization_id,
            placement_id=db_model.placement_id,
            pay_system=db_model.pay_system_id,
            wage_system=db_model.wage_system,
            position_id=db_model.position_id,
            position_name=db_model.position_name,
            position_type=db_model.position_type and PositionType(db_model.position_type),
            rate=db_model.rate,
            review_scheme_id=db_model.review_scheme_id,
            salary=db_model.salary,
            ticket=db_model.ticket,
            optional_ticket=db_model.optional_ticket,
            remove_budget_position=db_model.remove_budget_position,
            unit_manager=db_model.unit_manager,
            employment_type=db_model.employment_type,
            person_id=db_model.staff_id,
            physical_entity_id=db_model.person_id,
            other_payments=db_model.other_payments,
            join_at=db_model.join_at,
            probation_period_code=db_model.probation_period_code,
            is_replacement=db_model.is_replacement,
            instead_of_login=db_model.instead_of_login,
            contract_term_date=db_model.contract_term_date,
            contract_period=db_model.contract_period,
        )
        return res

    def _change_to_model(self, change: entities.Change) -> ChangeRegistry:
        geography_id = (
            Geography.objects.get(department_instance__url=change.geography_url).id
            if change.geography_url
            else None
        )
        result = ChangeRegistry(
            id=change.id,
            budget_position_id=change.budget_position and change.budget_position.id,
            new_budget_position_id=change.new_budget_position and change.new_budget_position.id,
            linked_budget_position_id=change.linked_budget_position_id,
            pushed_to_femida=change.pushed_to_femida,
            oebs_transaction_id=change.oebs_transaction_id,
            last_oebs_error=change.last_oebs_error,
            correction_id=change.correction_id,
            push_status=change.push_status,
            sent_to_oebs=change.sent_to_oebs,
            assignment_id=change.assignment_id,
            workflow_id=change.workflow_id,
            review_scheme_id=change.review_scheme_id,
            staff_bonus_scheme_id=change.bonus_scheme_id,
            compensation_scheme_id=change.reward_scheme_id,
            currency=change.currency,
            department_id=change.department_id,
            dismissal_date=change.dismissal_date,
            effective_date=change.effective_date,
            geo_id=geography_id,
            oebs_grade_id=change.grade_id,
            headcount=change.headcount,
            staff_hr_product_id=change.hr_product_id,
            office_id=change.office_id,
            organization_id=change.organization_id,
            placement_id=change.placement_id,
            pay_system_id=change.pay_system,
            wage_system=change.wage_system,
            position_id=change.position_id,
            position_name=change.position_name,
            position_type=change.position_type and change.position_type.value,
            rate=change.rate,
            salary=change.salary,
            staff_id=change.person_id,
            person_id=change.physical_entity_id,
            ticket=change.ticket,
            optional_ticket=change.optional_ticket,
            remove_budget_position=change.remove_budget_position,
            unit_manager=change.unit_manager,
            employment_type=change.employment_type,
            other_payments=change.other_payments,
            join_at=change.join_at,
            probation_period_code=change.probation_period_code,
            is_replacement=change.is_replacement,
            instead_of_login=change.instead_of_login,
            contract_term_date=change.contract_term_date,
            contract_period=change.contract_period,
        )

        if change.oebs_idempotence_key:  # to make django field defaults working
            result.oebs_idempotence_key = change.oebs_idempotence_key,
        return result

    def _create_workflow_from_db(self, workflow_model: Workflow) -> entities.AbstractWorkflow:
        workflow_class = self.workflows_by_code[workflow_model.code]
        changes = [
            self._change_from_model(change_model)
            for change_model in workflow_model.changeregistry_set.all().order_by('id')
        ]
        result: entities.AbstractWorkflow = workflow_class(workflow_model.id, changes)
        result.status = workflow_model.status
        result.manually_processed = workflow_model.manually_processed
        result.proposal_id = workflow_model.proposal_id
        result._proposal_meta = workflow_model.proposal
        result.vacancy_id = workflow_model.vacancy_id
        result.credit_management_id = workflow_model.credit_management_id
        result._credit_management = workflow_model.credit_management
        result.confirmed_at = workflow_model.confirmed_at
        result.catalyst_id = workflow_model.catalyst_id
        result.permission_date = workflow_model.permission_date
        return result

    @staticmethod
    def has_only_pending_workflows_for_credit_repayment(credit_management_id: int) -> bool:
        qs = (
            Workflow.objects
            .filter(credit_management_id=credit_management_id)
            .exclude(status=WORKFLOW_STATUS.PENDING)
        )
        return not qs.exists()
