from datetime import date
import logging
from typing import Callable
from uuid import UUID

from staff.budget_position.workflow_service import entities


logger = logging.getLogger(__name__)


class PushWorkflowToOebs:
    def __init__(
        self,
        repository: entities.WorkflowRepositoryInterface,
        oebs_service: entities.OEBSService,
        staff_service: entities.StaffService,
        update_push_status: Callable[[UUID], None],
    ):
        self._repository = repository
        self._oebs_service = oebs_service
        self._staff_service = staff_service
        self._update_push_status = update_push_status

    def push(self, workflow_id: UUID, person_id: int) -> None:
        workflow = self._repository.get_by_id(workflow_id)
        workflow.catalyst_id = person_id
        workflow.permission_date = date.today()

        if workflow.should_be_marked_manually_processed_automatically:
            logger.info('Workflow should be marked manually processed')
            workflow.mark_manually_processed(person_id)
            self._repository.save(workflow)
            return

        workflow.mark_pushing_to_oebs()
        logger.info('Marked workflow as pushing to oebs')

        person = self._staff_service.get_person(workflow.catalyst_id)
        self._oebs_service.push_next_change_to_oebs(workflow, person.login)
        self._repository.save(workflow)
        self._update_push_status(workflow_id)
