import logging
from typing import Callable
from uuid import UUID

from staff.budget_position.workflow_service import entities
from staff.budget_position.workflow_service.use_cases.interfaces import CreditManagementServiceInterface


logger = logging.getLogger(__name__)


class UpdateWorkflowPushStatus:
    def __init__(
        self,
        repository: entities.WorkflowRepositoryInterface,
        oebs_service: entities.OEBSService,
        update_push_status: Callable[[UUID], None],
        credit_management_service: CreditManagementServiceInterface,
        staff_service: entities.StaffService,
    ):
        self._update_push_status = update_push_status
        self._repository = repository
        self._oebs_service = oebs_service
        self._credit_management_service = credit_management_service
        self._staff_service = staff_service

    def update_status(self, workflow_id: UUID) -> None:
        logger.info('Trying to update workflow push status')
        workflow = self._repository.get_by_id(workflow_id)
        if workflow.is_finished():
            self._notify_credit_management_service(workflow)
        else:
            assert len(workflow.changes) > 0
            can_push_next_change = self._oebs_service.update_changes_push_status(workflow.changes)
            if can_push_next_change:
                person = self._staff_service.get_person(workflow.catalyst_id)
                try:
                    self._oebs_service.push_next_change_to_oebs(workflow, person.login)
                except entities.OEBSError:
                    self._update_push_status(workflow.id)
            else:
                self._update_push_status(workflow.id)
            self._repository.save(workflow)

    def _notify_credit_management_service(self, workflow: entities.AbstractWorkflow):
        if workflow.credit_management_id:
            self._credit_management_service.notify_about_changes_for_application(workflow.credit_management_id, None)
