import logging
from typing import List, Optional
from uuid import UUID

from lagom import Container

from staff.lib.db import atomic
from staff.lib.log import log_context

from staff.budget_position import const
from staff.budget_position.workflow_service import entities, use_cases
from staff.budget_position.workflow_service.container import build_container


logger = logging.getLogger(__name__)


class WorkflowRegistryService:
    """
    Facade for workflow registry
    """

    def __init__(self, container: Optional[Container] = None) -> None:
        self.container = container or build_container()

    @atomic
    def mark_workflow_processed_manually(self, workflow_id: UUID, person_id: int) -> None:
        repository = self.container[entities.WorkflowRepositoryInterface]
        with log_context(workflow_id=str(workflow_id)):
            workflow = repository.get_by_id(workflow_id)
            workflow.mark_manually_processed(person_id)
            logger.info('Marked workflow as processed manually')
            repository.save(workflow)

    @atomic
    def try_create_workflow_from_femida(self, data: entities.FemidaData) -> UUID:
        with log_context(
            vacancy_id=data.vacancy_id,
            ticket=data.ticket,
            expected_exceptions=[entities.OEBSError, entities.AbstractWorkflowResolveError],
        ):
            use_case = self.container[use_cases.CreateWorkflowFromFemida]
            return use_case.create(data)

    @atomic
    def try_create_workflow_from_proposal(
        self,
        data: List[entities.ProposalData],
    ) -> (List[UUID], List[entities.AbstractWorkflow]):
        use_case = self.container[use_cases.CreateWorkflowFromProposal]
        return use_case.create(data)

    @atomic
    def try_create_workflow_for_credit_repayment(self, data: entities.CreditRepaymentData) -> UUID:
        use_case = self.container[use_cases.CreateWorkflowForCreditRepayment]
        return use_case.create(data)

    @atomic
    def try_create_workflow_for_maternity(self, data: entities.MoveToMaternityData) -> UUID:
        use_case = self.container[use_cases.CreateWorkflowForMaternity]
        return use_case.create(data)

    def workflow_can_be_cancelled(self, workflow_id: UUID) -> bool:
        workflow_repository = self.container[entities.WorkflowRepositoryInterface]
        workflow = workflow_repository.get_by_id(workflow_id)
        return workflow.can_be_cancelled()

    @atomic
    def cancel_workflow(self, workflow_id: UUID, person_id: Optional[int]) -> None:
        with log_context(workflow_id=str(workflow_id)):
            use_case = self.container[use_cases.Cancel]
            use_case.cancel(workflow_id, person_id)

    @atomic
    def cancel_workflows_for_proposal(self, proposal_id: int) -> None:
        workflow_repository = self.container[entities.WorkflowRepositoryInterface]
        with log_context(proposal_id=proposal_id):
            logger.info('Cancelling workflows for proposal %s', proposal_id)
            workflow_repository.cancel_workflows_for_proposal(proposal_id)

    @atomic
    def cancel_workflows_for_credit_repayment(self, credit_management_application_id: int) -> None:
        with log_context(credit_management_id=credit_management_application_id):
            logger.info('Cancelling workflows for credit repayment %s', credit_management_application_id)
            use_case = self.container[use_cases.Cancel]
            use_case.cancel_workflows_for_credit_repayment(credit_management_application_id)

    @atomic
    def confirm_workflows_for_proposal(self, proposal_id: int) -> None:
        workflow_repository = self.container[entities.WorkflowRepositoryInterface]
        with log_context(proposal_id=proposal_id):
            logger.info('Confirming workflows for proposal %s', proposal_id)
            for workflow in workflow_repository.get_workflows_by_proposal_id(proposal_id):
                self.confirm_workflow(workflow.id, None)

    @atomic
    def confirm_workflows_for_credit_repayment(self, credit_management_application_id: int) -> None:
        with log_context(credit_repayment_application_id=credit_management_application_id):
            logger.info('Confirming and pushing for credit repayment %s', credit_management_application_id)
            use_case = self.container[use_cases.ConfirmWorkflows]
            use_case.confirm_for_credit_management_application(credit_management_application_id)

    @atomic
    def confirm_workflow(self, workflow_id: UUID, responsible_id: Optional[int]):
        with log_context(workflow_id=str(workflow_id)):
            use_case = self.container[use_cases.ConfirmWorkflows]
            use_case.confirm(workflow_id, responsible_id)

    @atomic
    def confirm_workflows(self, workflows_ids: List[UUID]):
        with log_context(workflow_ids=str(workflows_ids)):
            use_case = self.container[use_cases.ConfirmWorkflows]
            use_case.confirm_many(workflows_ids)

    @atomic
    def try_update_workflow_push_status(self, workflow_id: UUID) -> None:
        with log_context(workflow_id=str(workflow_id)):
            try:
                use_case = self.container[use_cases.UpdateWorkflowPushStatus]
                use_case.update_status(workflow_id)
            except entities.OEBSError:
                logger.info('OEBS error', exc_info=True)
            except Exception:
                logger.exception('Unhandled exception')
                raise

    @atomic
    def push_workflow_to_oebs(self, workflow_id: UUID, person_id: int) -> None:
        with log_context(workflow_id=str(workflow_id), expected_exceptions=[entities.OEBSError]):
            assert person_id is not None
            use_case = self.container[use_cases.PushWorkflowToOebs]
            use_case.push(workflow_id, person_id)

    @atomic
    def push_failed_workflows_to_oebs(self, workflow_ids: List[UUID], person_id: int) -> None:
        workflow_repository = self.container[entities.WorkflowRepositoryInterface]
        workflow_repository.retry_workflows(workflow_ids, person_id)

    @atomic
    def push_proposal_to_oebs(self, proposal_id: int) -> None:
        workflow_repository = self.container[entities.WorkflowRepositoryInterface]
        wfs = workflow_repository.get_workflows_by_proposal_id(proposal_id, status=const.WORKFLOW_STATUS.CONFIRMED)
        wf_ids = [it.id for it in wfs]
        workflow_repository.queue_workflows(wf_ids, None)

    def has_only_pending_workflows_for_credit_repayment(self, credit_management_id: int) -> bool:
        workflow_repository = self.container[entities.WorkflowRepositoryInterface]
        return workflow_repository.has_only_pending_workflows_for_credit_repayment(credit_management_id)
