from django.conf import settings

from staff.person.models import Staff

from staff.card_order.builders import NativeCardBuilder, EnglishCardBuilder
from staff.card_order.objects import Card, CardTicket, CardTicketException
from staff.lib.forms.staff_form_grid import StaffFormGrid
from staff.lib.exceptions import ErrorWithStatusCode
from .forms import OrderCardForm, OrderCardEnForm, OrderCardBackForm

import logging

logger = logging.getLogger('card_order.card')


class CardOrderError(ErrorWithStatusCode):
    pass


def get_initial_card(login):
    try:
        person = Staff.objects.get(login=login)
    except Staff.DoesNotExist:
        logger.exception('Error looking for person with login %s', login)
        raise CardOrderError('card-order_person-doesnt-exist', 404)

    card = Card(person)

    return {
        'native': StaffFormGrid(
            OrderCardForm, initial=[card.native_data]).as_front_dict(),
        'english': StaffFormGrid(
            OrderCardEnForm, initial=[card.english_data]).as_front_dict(),
        'back': StaffFormGrid(
            OrderCardBackForm, initial=[card.english_data]).as_front_dict(),
        'choices': OrderCardForm().choices_as_front_dict(),
    }


def _get_person(target_login):
    try:
        return Staff.objects.get(login=target_login)
    except Staff.DoesNotExist:
        logger.exception('Error looking for person with login %s', target_login)
        raise CardOrderError('card-order_person-doesnt-exist', 404)


def _create_ticket(is_single, pdf, person, author, office):
    try:
        ticket = CardTicket(
            is_single=is_single, staff=person, creator=author, office=office)
        ticket.create(pdf)
    except Exception as e:
        if isinstance(e, CardTicketException):
            logger.exception('')
        else:
            logger.exception(
                'unhandled exception while ordering cards for %s', person.login
            )
        if settings.DEBUG:
            raise
        raise CardOrderError('card-order_general-error', 500)
    else:
        logger.info('Created ticket %s', ticket.issue['key'])
        return {
            'ticket_url': ticket.url,
            'ticket': ticket.issue['key']
        }


def get_card_builder(lang):
    return NativeCardBuilder() if lang == 'native' else EnglishCardBuilder()


def order(target_login, office, pdf, is_single, author):
    person = _get_person(target_login)
    return _create_ticket(is_single, pdf, person, author, office)
