import json

from staff.lib import requests

from django.conf import settings
from django.template import loader, Context
from django.utils.text import capfirst

from ids.configuration import get_config

from staff.person.models import DOMAIN

import logging

logger = logging.getLogger('card_order')


UPLOAD_ATTACHMENT_TIMEOUT = 7
CREATE_TICKET_TIMEOUT = 8


class CardTicketException(Exception):
    pass


class RequestTimeoutError(CardTicketException):
    pass


class STAuthorizationError(CardTicketException):
    pass


class STQueueNotFound(CardTicketException):
    pass


def _cut_braces(phone):
    return str(phone).replace('(', '').replace(')', '')


class Card(object):

    def __init__(self, staff):
        super(Card, self).__init__()
        self.staff = staff

    def _data(self):
        _data = {
            'office': self.staff.office,
            'work_phone': self.staff.work_phone,
            'mobile_phone': _cut_braces(self.staff.mobile_phone),
            'extra_phone': '',
            'show_fax': 'false',
            'other': '',
        }

        if self.staff.email:
            _data.update({
                'email_login': self.staff.email.split('@')[0],
                'email_domain': DOMAIN.get_key(self.staff.email.split('@')[1])
            })
        return _data

    @property
    def native_data(self):
        data = {
            'last_name': self.staff.last_name,
            'first_name': self.staff.first_name,
            'middle_name': self.staff.middle_name,
            'position': capfirst(self.staff.position),
        }
        data.update(self._data())
        return data

    @property
    def english_data(self):
        data = {
            'last_name': self.staff.last_name_en,
            'first_name': self.staff.first_name_en,
            'middle_name': '',
            'position': capfirst(self.staff.position_en),
        }
        data.update(self._data())
        return data


class CardTicket(object):
    template = 'card.wiki'

    def __init__(self, is_single, staff, creator, office):
        super(CardTicket, self).__init__()
        self.is_single = is_single
        self.staff = staff
        self.creator = creator
        self.office = office
        self.issue = None
        self.session = None

    def render(self):
        context = {
            'staff': self.staff,
            'office': self.office,
            'creator': self.creator,
            'is_single': self.is_single
        }
        template = loader.get_template(self.template)
        content = template.render(Context(context))
        subject, content_html = content.split('\n\n', 1)
        return content_html, subject

    def upload_attachment(self, pdf):
        url = self.startrek_api_url(resource='attachments')
        try:
            response = self.session.post(
                url,
                files={'file': (
                    '{login}.pdf'.format(login=self.staff.login),
                    pdf,
                    'application/pdf',
                    {'Expires': '0'}
                )},
                timeout=UPLOAD_ATTACHMENT_TIMEOUT,
            )
        except requests.Timeout:
            raise RequestTimeoutError('Timeout exceeded while uploading '
                                      'file to startrek')
        if response.status_code == 401:
            raise STAuthorizationError(
                'Bad authorization while uploading file to startrek'
                'response %s' % response.content
            )
        return response.json().get('id')

    def create_startrek_ticket(self, summary, description, attachment_id):
        url = self.startrek_api_url(resource='issues')
        self.session.headers['Content-Type'] = 'application/json'
        params = {
            'createdBy': self.creator.login,
            'queue': settings.CARD_ORDER_PROJECT,
            'summary': summary,
            'description': description,
            'components': settings.CARD_ORDER_COMPONENT,
            'type': settings.CARD_ORDER_TYPE,
            'attachmentIds': [attachment_id],
        }
        try:
            response = self.session.post(url, data=json.dumps(params),
                                         timeout=CREATE_TICKET_TIMEOUT)
        except requests.Timeout:
            raise RequestTimeoutError('Timeout exceeded while uploading '
                                      'file to startrek')
        if response.status_code == 404:
            raise STQueueNotFound(
                'Queue with key %s was not found! Response: %s' %
                (settings.CARD_ORDER_PROJECT, response.content)
            )

        return response.json()

    def create(self, pdf):
        token = settings.ROBOT_STAFF_OAUTH_TOKEN

        self.session = requests.Session()
        self.session.headers['Authorization'] = 'OAuth %s' % token

        attachment_id = self.upload_attachment(pdf)
        description, summary = self.render()
        self.issue = self.create_startrek_ticket(
            summary,
            description,
            attachment_id
        )

    def startrek_api_url(self, resource):
        url = '{protocol}://{host}/v2/{resource}'
        return url.format(resource=resource, **get_config('STARTREK_API'))

    @property
    def url(self):
        st = get_config('STARTREK')
        return '{protocol}://{host}/{key}'.format(
            protocol=st['protocol'],
            host=st['host'],
            key=self.issue['key'],
        )
