from rest_framework.response import Response
from rest_framework.views import APIView

from .domain import (
    list_tasks,
    list_running,
    run_task,
    get_result,
    list_finished,
)
from .permissions import EditTaskPermission
from .serializers import TaskSerializer, ResultSerializer


class CeleryToolsAPIView(APIView):
    permission_classes = (EditTaskPermission,)


class TaskListView(CeleryToolsAPIView):
    """List all known tasks"""

    def get(self, request, format=None):
        tasks = list_tasks()
        serializer = TaskSerializer(tasks, many=True)
        return Response(serializer.data)


class FinishedTaskListView(CeleryToolsAPIView):
    """List all finished tasks"""

    def get(self, request, format=None):
        tasks = list_finished()
        return Response(tasks)


class RunningTaskListView(CeleryToolsAPIView):
    """List all running tasks or launch a new one"""

    def get(self, request, format=None):
        tasks = list_running()
        return Response(tasks)

    def post(self, request, format=None):
        name = request.DATA['name']
        queue = request.DATA.get('queue')
        async_result = run_task(name, (), {}, queue=queue)
        return Response({'name': async_result.task_name,
                         'id': async_result.task_id})


class RunningTaskDetailsView(CeleryToolsAPIView):
    """Show details of a running task or cancel it"""

    def get(self, request, task_id, format=None):
        async_result = get_result(task_id)
        serializer = ResultSerializer(async_result)
        return Response(serializer.data)
