import json
from typing import Dict, List

from collections import namedtuple
from datetime import datetime

from factory.django import mute_signals
from mongomock import MongoClient
import pytest
from pytest_django.fixtures import SettingsWrapper

from django.conf import settings
from django.contrib.auth.models import Permission
from django.db.models import signals
from django.core.urlresolvers import reverse
from django.test import RequestFactory

from staff.achievery.tests.factories.model import AchievementFactory, IconFactory
from staff.departments.models import DEADLINE_TYPE
from staff.departments.tests.factories import HrDeadlineFactory, VacancyFactory
from staff.femida.constants import VACANCY_STATUS
from staff.gap.api.forms import PeriodicType
from staff.gap.controllers.counter import CounterCtl
from staff.gap.controllers.gap import GapCtl, PeriodicGapCtl
from staff.gap.controllers.templates import TemplatesCtl
from staff.gap.tests.constants import (
    COUNTERS_MONGO_COLLECTION,
    GAPS_MONGO_COLLECTION,
    TEMPLATES_MONGO_COLLECTION,
    PERIODIC_GAPS_MONGO_COLLECTION,
)
from staff.lib.auth.utils import get_or_create_test_user
from staff.lib.db import atomic
from staff.lib.mongodb import mongo
from staff.lib.testing import factory, FloorFactory, OfficeFactory, RouteFactory, StaffFactory, TableFactory
from staff.lib.tests.pytest_fixtures import AttrDict, create_company, create_kinds, create_map_models


__NOTIFICATIONS = []

NotificationSendCall = namedtuple('NotificationSendCall', ['notification', 'params'])


@pytest.fixture
def notifications():
    """Содержит список именованных кортежей NotificationSendCall по количеству
    запросов на отправку оповещений.

    Каждый кортеж содержит объект оповещения и параметры вызова его .send().

    :rtype: list[NotificationSendCall]
    """
    return __NOTIFICATIONS


@pytest.fixture(autouse=True)
def notification_send(monkeypatch):
    """Подключается автоматически, заменяет вызов метода отправки
    оповещения на имитатор, запоминающий отправления для возможности
    последующего доступа к ним через фикстуру notifications.

    """
    from staff.django_intranet_notifications import Notification

    monkeypatch.setattr(
        Notification,
        'send',
        lambda self, **params: __NOTIFICATIONS.append(NotificationSendCall(self, params))
    )

    try:
        yield

    finally:
        __NOTIFICATIONS[:] = []


@pytest.fixture()
def map_models(db) -> AttrDict:
    return create_map_models()


@pytest.fixture
def kinds(db):
    return create_kinds()


@pytest.fixture()
def company(db, kinds, map_models, settings) -> AttrDict:
    return create_company(kinds, map_models, settings)


@pytest.yield_fixture(scope='module')
def settings_with_module_scope(django_db_blocker):
    with django_db_blocker.unblock():
        wrapper = SettingsWrapper()

    yield wrapper
    wrapper.finalize()


@pytest.yield_fixture(scope='module')
def company_with_module_scope(django_db_blocker, settings_with_module_scope):
    django_db_blocker.unblock()
    try:
        with atomic():
            kinds = create_kinds()
            map_models = create_map_models()
            result = create_company(kinds, map_models, settings_with_module_scope)
            django_db_blocker.block()

            try:
                yield result
            finally:
                django_db_blocker.unblock()
                raise InterruptedError()  # recommended way to rollback changes

    except InterruptedError:
        pass
    finally:
        django_db_blocker.block()


@pytest.fixture
def mocked_mongo(monkeypatch):
    monkeypatch.setattr(mongo, '_db', MongoClient().db)
    return mongo


@pytest.fixture
def deadlines():
    return {
        type_: HrDeadlineFactory(type=type_)
        for type_, _ in DEADLINE_TYPE.choices()
    }


@pytest.fixture
def vacancies(company, db):
    deps = company.departments
    result = [
        VacancyFactory(
            department=deps['yandex'],
            is_published=True,
            status=VACANCY_STATUS.IN_PROGRESS,
        ),
        VacancyFactory(
            department=deps['yandex'],
            is_published=True,
            status=VACANCY_STATUS.IN_PROGRESS,
        ),
        VacancyFactory(
            department=deps['yandex_dep2'],
            is_published=True,
            status=VACANCY_STATUS.IN_PROGRESS,
        ),
        VacancyFactory(
            department=deps['yandex_dep2'],
            is_published=True,
            status=VACANCY_STATUS.IN_PROGRESS,
        ),
        VacancyFactory(
            department=deps['yandex_dep1_dep12'],
            is_published=True,
            status=VACANCY_STATUS.IN_PROGRESS,
        ),
        VacancyFactory(
            department=deps['yandex_dep1_dep12'],
            is_published=True,
            status=VACANCY_STATUS.IN_PROGRESS,
        ),
        VacancyFactory(
            department=deps['yandex_dep1_dep11_dep111'],
            is_published=True,
            status=VACANCY_STATUS.OFFER_PROCESSING,
        ),
        VacancyFactory(
            department=deps['yandex_dep1_dep11_dep111'],
            is_published=True,
            status=VACANCY_STATUS.OFFER_PROCESSING,
        ),

        VacancyFactory(
            department=deps['yandex_dep1_dep11_dep111'],
            is_published=True,
            status=VACANCY_STATUS.ON_APPROVAL,
        ),
        VacancyFactory(
            department=deps['yandex_dep1_dep11_dep111'],
            is_published=True,
            status=VACANCY_STATUS.OFFER_ACCEPTED,
        ),
        VacancyFactory(
            department=deps['yandex_dep1_dep11_dep111'],
            is_published=False,
            status=VACANCY_STATUS.OFFER_PROCESSING,
        ),
    ]
    return {v.id: v for v in result}


@pytest.fixture
def superuser_client(client):
    with factory.django.mute_signals(signals.post_save):
        tester = get_or_create_test_user()
        tester.is_superuser = True
        tester.save()

    return client


@pytest.fixture
def fetcher(client):

    with factory.django.mute_signals(signals.post_save):
        tester = get_or_create_test_user()
        tester.save()

    class PermittedFetcher(object):
        """ Для запросов с указанными правами """
        def __init__(self, fetcher_client):
            self.client = fetcher_client
            self.tester = tester

        def get(self, *args, **kwargs):
            permissions = kwargs.get('permissions', [])
            for perm in Permission.objects.filter(codename__in=permissions):
                self.tester.user_permissions.add(perm)

            result = self.client.get(*args, **kwargs)

            for perm in Permission.objects.filter(codename__in=permissions):
                self.tester.user_permissions.remove(perm)

            return result

        def post(self, *args, **kwargs):
            permissions = kwargs.get('permissions', [])
            for perm in Permission.objects.filter(codename__in=permissions):
                self.tester.user_permissions.add(perm)

            result = self.client.post(*args, **kwargs)

            for perm in Permission.objects.filter(codename__in=permissions):
                self.tester.user_permissions.remove(perm)

            return result

    return PermittedFetcher(client)


@pytest.fixture
def robot_staff_user():
    s = StaffFactory(login=settings.ROBOT_STAFF_LOGIN)
    s.user.username = s.login
    s.user.save()

    return s.user


@pytest.fixture
def achievements(robot_staff_user):
    assert settings.ACHIEVERY_ROBOT_PERSON_LOGIN == robot_staff_user.username
    beginner_ach = AchievementFactory(id=settings.ACHIEVEMENT_BEGINNER_ID)
    restored_ach = AchievementFactory(id=settings.ACHIEVEMENT_RESTORED_ID)
    employee_ach = AchievementFactory(id=settings.ACHIEVEMENT_EMPLOYEE_ID)
    IconFactory(achievement=beginner_ach, level=-1)
    IconFactory(achievement=restored_ach, level=-1)
    IconFactory(achievement=employee_ach, level=1)
    RouteFactory(transport_id='email', params='{}')


class MapTestData(object):
    redrose = None
    comode = None
    first = None
    second = None
    tbl_1 = None
    tbl_2 = None
    jay = None
    bob = None


@pytest.fixture
@mute_signals(signals.pre_save, signals.post_save)
def map_test_data():
    result = MapTestData()
    result.redrose = OfficeFactory(name='Red Rose', city=None, intranet_status=1)
    result.comode = OfficeFactory(name='Comode', city=None, intranet_status=1)

    result.first = FloorFactory(name='First', office=result.redrose, intranet_status=1)
    result.second = FloorFactory(name='Second', office=result.comode, intranet_status=1)

    result.tbl_1 = TableFactory(
        floor=result.first,
        intranet_status=1,
        coord_x=100,
        coord_y=100,
        num=1,
    )
    result.tbl_2 = TableFactory(
        floor=result.second,
        intranet_status=1,
        coord_x=200,
        coord_y=200,
        num=2,
    )
    result.tbl_with_coord_is_0 = TableFactory(
        floor=result.second,
        intranet_status=1,
        coord_x=0,
        coord_y=200,
        num=3,
    )

    result.jay = StaffFactory(
        login='jay',
        is_dismissed=False,
        table=result.tbl_1,
        department=None,
        organization=None,
    )
    result.bob = StaffFactory(
        login='bob',
        is_dismissed=True,
        table=result.tbl_1,
        department=None,
        organization=None,
    )

    return result


class GapTest(object):
    DEFAULT_GAP_ID = 1

    dep_yandex = None
    test_person = None
    DEFAULT_MODIFIER_ID = None

    def get_base_gap(self, workflow_cls):
        return {
            'workflow': workflow_cls.workflow,
            'date_from': datetime(2015, 1, 1, 10, 20),
            'date_to': datetime(2015, 1, 2, 11, 30),
            'full_day': True,
            'work_in_absence': True,
            'created_by_uid': self.test_person.uid,
            'person_id': self.test_person.id,
            'person_login': self.test_person.login,
            'comment': '',
            'table': self.test_table.id,
            'to_notify': [],
        }

    def get_base_periodic_gap(self, workflow_cls):
        return {
            'workflow': workflow_cls.workflow,
            'date_from': datetime(2015, 1, 1, 0, 0),
            'date_to': datetime(2015, 1, 1, 0, 0),
            'office': 1,
            'full_day': True,
            'work_in_absence': True,
            'created_by_uid': self.test_person.uid,
            'person_id': self.test_person.id,
            'person_login': self.test_person.login,
            'comment': '',
            'to_notify': [],
            'table': self.test_table.id,
            'is_periodic_gap': True,
            'periodic_date_to': datetime(2015, 1, 20, 0, 0),
            'periodic_type': PeriodicType.WEEK.value,
            'period': 1,
            'periodic_map_weekdays': [3, 5],
        }

    def get_api_base_create_gap(self, workflow_cls):
        return {
            'workflow': workflow_cls.workflow,
            'date_from': datetime(2015, 1, 1, 10, 20),
            'date_to': datetime(2015, 1, 2, 11, 30),
            'full_day': True,
            'work_in_absence': True,
            'person_login': self.test_person.login,
            'comment': '',
            'to_notify': [],
        }

    def get_api_base_edit_gap(self, gap_id):
        return {
            'gap_id': gap_id,
            'date_from': datetime(2016, 1, 1, 10, 20),
            'date_to': datetime(2016, 1, 2, 11, 30),
            'full_day': False,
            'work_in_absence': False,
            'comment': 'test',
            'to_notify': ['email0@test.com', 'email1@test.com'],
        }

    def mongo_now(self):
        return datetime.now().replace(microsecond=0)  # привет Монге

    def base_assert_new_gap(self, workflow_cls, now, src_gap, dst_gap):
        assert dst_gap is not None
        assert 'log' in dst_gap

        assert dst_gap['workflow'] == workflow_cls.workflow
        assert dst_gap['gap_type'] == workflow_cls.gap_type
        assert dst_gap['person_id'] == src_gap['person_id']
        assert dst_gap['person_login'] == src_gap['person_login']
        assert dst_gap['work_in_absence'] == src_gap['work_in_absence']
        assert dst_gap['full_day'] == src_gap['full_day']
        if not dst_gap.get('periodic_gap_id'):
            assert dst_gap['date_from'] == src_gap['date_from']
            assert dst_gap['date_to'] == src_gap['date_to']
        assert dst_gap['comment'] == ''
        assert dst_gap['created_by_uid'] == src_gap['created_by_uid']

        if dst_gap['workflow'] != 'absence':
            assert dst_gap['to_notify'] == []

        assert dst_gap['created_at'] >= now
        assert dst_gap['modified_at'] >= now


@pytest.yield_fixture
def gap_test(mocked_mongo, company, map_test_data):
    CounterCtl.MONGO_COLLECTION = COUNTERS_MONGO_COLLECTION
    GapCtl.MONGO_COLLECTION = GAPS_MONGO_COLLECTION
    PeriodicGapCtl.MONGO_COLLECTION = PERIODIC_GAPS_MONGO_COLLECTION
    TemplatesCtl.MONGO_COLLECTION = TEMPLATES_MONGO_COLLECTION

    CounterCtl().set_counter(GapCtl.COUNTER_NAME, 0, GapCtl.MONGO_COLLECTION)
    CounterCtl().set_counter(PeriodicGapCtl.COUNTER_NAME, 0, PeriodicGapCtl.MONGO_COLLECTION)
    GapCtl().recreate_collection()
    PeriodicGapCtl().recreate_collection()
    TemplatesCtl().recreate_collection()

    result = GapTest()
    result.dep_yandex = company.yandex

    result.test_person = StaffFactory(
        department=result.dep_yandex,
        login='test_person',
        office=company['offices']['KR'],
        uid='1120000000018264',
    )

    result.test_table = map_test_data.tbl_1

    result.DEFAULT_MODIFIER_ID = result.test_person.id

    yield result

    mongo.db.drop_collection(COUNTERS_MONGO_COLLECTION)
    mongo.db.drop_collection(GAPS_MONGO_COLLECTION)
    mongo.db.drop_collection(PERIODIC_GAPS_MONGO_COLLECTION)
    mongo.db.drop_collection(TEMPLATES_MONGO_COLLECTION)


@pytest.fixture
def post_rf(rf: RequestFactory):
    def f(view_name: str, json_data: Dict = None, user=None):
        request = rf.post(
            reverse(view_name),
            data=json.dumps(json_data),
            content_type='application/json',
        )

        if user:
            request.user = user

        return request

    return f


@pytest.fixture
def get_rf(rf: RequestFactory):
    def f(view_name: str, reverse_args: List = None, params: Dict = None, user=None):
        request = rf.get(reverse(view_name, args=reverse_args), data=params)

        if user:
            request.user = user

        return request

    return f
