import attr
from datetime import date
from typing import List, Optional, Union


@attr.s(auto_attribs=True)
class PersonAction:

    @attr.s(auto_attribs=True)
    class DepartmentChange:
        changing_duties: Optional[bool] = None
        department: Optional[str] = None
        fake_department: Optional[str] = None
        from_maternity_leave: Optional[bool] = None
        service_groups: Optional[List[str]] = None
        vacancy_url: Optional[str] = None
        with_budget: Optional[bool] = None

    @attr.s(auto_attribs=True)
    class PositionChange:
        new_position: Optional[str] = None
        position_legal: Union[str, int, None] = None

    @attr.s(auto_attribs=True)
    class GradeChange:
        new_grade: Optional[str] = None
        occupation: Optional[int] = None
        force_recalculate_schemes: bool or None = None

    @attr.s(auto_attribs=True)
    class SalaryChange:
        new_currency: Optional[str] = None
        new_rate: Optional[str] = None
        new_salary: Optional[str] = None
        new_wage_system: Optional[str] = None
        old_currency: Optional[str] = None
        old_rate: Optional[str] = None
        old_salary: Optional[str] = None
        old_wage_system: Optional[str] = None

    @attr.s(auto_attribs=True)
    class ValueStreamChange:
        value_stream: Optional[str] = None

    @attr.s(auto_attribs=True)
    class GeographyChange:
        geography: Optional[str] = None

    login: str
    sections: Optional[List[str]] = None
    department: DepartmentChange = attr.Factory(DepartmentChange)
    position: PositionChange = attr.Factory(PositionChange)
    salary: SalaryChange = attr.Factory(SalaryChange)
    grade: GradeChange = attr.Factory(GradeChange)
    value_stream: ValueStreamChange = attr.Factory(ValueStreamChange)
    geography: GeographyChange = attr.Factory(GeographyChange)
    action_id: Optional[str] = None
    comment: Optional[str] = None
    office: Optional[int] = None
    organization: Optional[int] = None
    dep_chain: Optional[List[str]] = None
    oebs_application_date: Optional[date] = None

    @staticmethod
    def _as_dict_filter_none(obj, recurse=True):
        return attr.asdict(obj, filter=lambda _, value: value is not None, recurse=recurse)

    def has_position_change(self):
        return bool(self._as_dict_filter_none(self.position))

    def has_dep_change(self):
        return bool(self._as_dict_filter_none(self.department))

    def has_salary_change(self):
        return bool(self._as_dict_filter_none(self.salary))

    def has_grade_change(self):
        return self.grade and self.grade.new_grade

    def has_valuestream_change(self):
        return self.value_stream and self.value_stream.value_stream

    def has_geography_change(self):
        return self.geography and self.geography.geography

    @classmethod
    def from_dict(cls, person_action):
        simple_fields = ['login', 'sections', 'action_id', 'comment']
        return cls(
            salary=PersonAction.SalaryChange(**person_action.get('salary', {})),
            department=PersonAction.DepartmentChange(**person_action.get('department', {})),
            position=PersonAction.PositionChange(**person_action.get('position', {})),
            grade=PersonAction.GradeChange(**person_action.get('grade', {})),
            value_stream=PersonAction.ValueStreamChange(**person_action.get('value_stream', {})),
            geography=PersonAction.GeographyChange(**person_action.get('geography', {})),
            organization=person_action.get('organization', {}).get('organization'),
            office=person_action.get('office', {}).get('office'),
            dep_chain=person_action.get('__department_chain__'),
            **{f: person_action.get(f) for f in simple_fields}
        )

    def copy_with_meta(self, **kwargs):
        return PersonAction(
            login=self.login,
            dep_chain=self.dep_chain,
            comment=self.comment,
            sections=self.sections,
            **kwargs
        )

    def as_dict(self):
        res = self._as_dict_filter_none(self, recurse=False)
        dep_chain = res.pop('dep_chain', None)

        if dep_chain is not None:
            res['__department_chain__'] = dep_chain

        for field in ('department', 'position', 'salary', 'value_stream', 'geography', 'grade'):
            if field not in res:
                continue
            as_dict = self._as_dict_filter_none(res[field])
            if as_dict:
                res[field] = as_dict
            else:
                del res[field]

        if 'office' in res:
            res['office'] = {'office': res['office']}

        if 'organization' in res:
            res['organization'] = {'organization': res['organization']}

        return res
