from datetime import datetime, date

from staff.lib import waffle
from mptt.exceptions import InvalidMove

from django.db import IntegrityError
from django.db.models import Q

from staff.departments.models import (
    Department,
    DepartmentStaff,
    DepartmentRoles,
    Vacancy,
)
from staff.person.models import Staff
from staff.groups.models import Group, GROUP_TYPE_CHOICES

from staff.audit.factory import create_log
from staff.headcounts.models import AllowedHeadcountOverdraft
from staff.person.controllers import PersonCtl

from staff.departments.utils import get_last_position
from staff.departments.edit.notifications import (
    send_department_created,
    send_department_deleted,
    send_chief_updated,
    send_deputies_updated,
)

from .exceptions import DepartmentCtlError
from .fields import CtlField, CtlCustomField
from .effects import (
    actualize_group,
    validate_parent_is_not_child,
)


class DepartmentCtl(object):
    CHIEF_ATTR = '_cached_chief_value'
    DEPUTIES_ATTR = '_cached_deputies_value'
    IMMUTABLE_FIELDS = {'pk', 'id', 'created_at', 'tree_id', 'lft', 'rght', 'level', 'parent_id'}
    ALL_FIELDS = {f.name for f in Department._meta.fields} | {'chief', 'deputies'}
    EDITABLE_FIELDS = ALL_FIELDS - IMMUTABLE_FIELDS

    def __init__(self, department, author_user=None):

        self._author = author_user  # Отсюда автор попадёт и в редактирование chief, deputies
        if isinstance(department, int):
            department = Department.objects.get(intranet_status=1, id=department)
        self.instance = department
        self.CTL_FIELDS = [
            f for f in self.ALL_FIELDS if f in DepartmentCtl.__dict__
        ]
        self._delayed_effects = []

    def __str__(self):
        return 'DepartmentCtl<%s>' % self.instance.name

    @staticmethod
    def comma_separated(values_list):
        if isinstance(values_list, str):
            return values_list
        return ','.join([v.strip() for v in values_list])

    @staticmethod
    def values_list(comma_separated):
        return [v.strip() for v in comma_separated.split(',')]

    def _get_chief(self):
        try:
            return DepartmentStaff.objects.get(
                department=self.instance,
                role_id=DepartmentRoles.CHIEF.value,
            ).staff
        except DepartmentStaff.DoesNotExist:
            return None

    def _set_chief(self, new_chief):
        try:
            role = DepartmentStaff.objects.get(
                department=self.instance,
                role_id=DepartmentRoles.CHIEF.value,
            )
        except DepartmentStaff.DoesNotExist:
            role = None

        old_chief = role and role.staff
        if old_chief == new_chief:
            return

        if new_chief:
            if role:
                role.staff = new_chief
                role.save()
            else:
                DepartmentStaff.objects.create(
                    staff=new_chief,
                    department=self.instance,
                    role_id=DepartmentRoles.CHIEF.value,
                )
        elif role:
            role.delete()
        if old_chief:
            PersonCtl(old_chief).actualize_is_big_boss().save(self._author)
        if new_chief:
            PersonCtl(new_chief).actualize_is_big_boss().save(self._author)

        send_chief_updated(self.instance, old_chief, new_chief)

    def _get_deputies(self):
        try:
            person_ids = DepartmentStaff.objects.filter(
                department=self.instance,
                role_id=DepartmentRoles.DEPUTY.value,
            ).values_list('staff', flat=True)

            return list(
                Staff.objects.filter(
                    is_dismissed=False,
                    id__in=person_ids,
                )
            )

        except DepartmentStaff.DoesNotExist:
            return []

    def _set_deputies(self, new_deputies):
        try:
            old_deputies = Staff.objects.filter(
                departmentstaff__role_id=DepartmentRoles.DEPUTY.value,
                departmentstaff__department=self.instance
            )
        except Staff.DoesNotExist:
            old_deputies = {}

        missing_deputies = set(new_deputies) - set(old_deputies)
        extra_deputies = set(old_deputies) - set(new_deputies)

        # Можно идти по zip_longest(old, new) и делать
        # force_update в DepartmentStaff( old ), подменяя ему staff
        for person in extra_deputies:
            DepartmentStaff.objects.get(
                department=self.instance,
                staff=person,
                role_id=DepartmentRoles.DEPUTY.value,
            ).delete()

        for person in missing_deputies:
            DepartmentStaff(
                department=self.instance,
                staff=person,
                role_id=DepartmentRoles.DEPUTY.value,
            ).save(force_insert=True)
        if extra_deputies or missing_deputies:
            send_deputies_updated(self.instance, old_deputies, new_deputies)

    def _get_maillists(self):
        return self.values_list(self.instance.maillists)

    def _set_maillists(self, new_maillists):
        self.instance.maillists = self.comma_separated(new_maillists)

    def _get_clubs(self):
        return self.values_list(self.instance.clubs)

    def _set_clubs(self, new_clubs):
        self.instance.clubs = self.comma_separated(new_clubs)

    def _get_headcount_overdraft_percents_with_childs(self):
        if not hasattr(self.instance, 'allowedheadcountoverdraft'):
            return None

        return self.instance.allowedheadcountoverdraft.percents_with_child_departments

    def _has_allowed_headcount_overdraft(self):
        return hasattr(self.instance, 'allowedheadcountoverdraft')

    def _set_headcount_overdraft_percents_with_childs(self, new_value):
        if not self._has_allowed_headcount_overdraft():
            #  To avoid reloading from db, separate creation and assignment
            overdraft_instance = AllowedHeadcountOverdraft.objects.create(department_id=self.instance.id)
            self.instance.allowedheadcountoverdraft = overdraft_instance

        self.instance.allowedheadcountoverdraft.percents_with_child_departments = new_value

    # todo: подумать нужно ли полностью заменить эти дескрипторы декоратором @property
    chief = CtlCustomField(
        'chief',
        getter_func=_get_chief,
        setter_func=_set_chief,
        cache_attr=CHIEF_ATTR,
    )

    deputies = CtlCustomField(
        'deputies',
        getter_func=_get_deputies,
        setter_func=_set_deputies,
        cache_attr=DEPUTIES_ATTR,
    )

    kind = CtlField('kind')
    name = CtlField('name', delayed_effects=[actualize_group])
    name_en = CtlField('name_en')
    maillists = CtlCustomField(
        'maillists',
        getter_func=_get_maillists,
        setter_func=_set_maillists,
        lazy_update=False,
    )

    clubs = CtlCustomField(
        'clubs',
        getter_func=_get_clubs,
        setter_func=_set_clubs,
        lazy_update=False,
    )

    parent = CtlField(
        'parent',
        validators=[validate_parent_is_not_child],
        delayed_effects=[actualize_group],
    )

    category = CtlField('category')

    allowed_overdraft_percents = CtlCustomField(
        'headcount_overdraft_percents_with_childs',
        getter_func=_get_headcount_overdraft_percents_with_childs,
        setter_func=_set_headcount_overdraft_percents_with_childs,
        lazy_update=False,
    )

    def population(self, direct=False):
        if not direct:
            persons = Staff.objects.filter(
                is_dismissed=False,
                department__lft__gte=self.instance.lft,
                department__rght__lte=self.instance.rght,
                department__tree_id=self.instance.tree_id,
            )
        else:
            persons = Staff.objects.filter(
                is_dismissed=False,
                department=self.instance,
            )
        return persons

    def vacancies(self, direct=False):
        if not direct:
            vacancies = Vacancy.objects.filter(
                is_active=True,
                department__lft__gte=self.instance.lft,
                department__rght__lte=self.instance.rght,
                department__tree_id=self.instance.tree_id,
            )
        else:
            vacancies = Vacancy.objects.filter(
                is_active=True,
                department=self.instance,
            )
        return vacancies

    @property
    def children_departments(self):
        return Department.objects.filter(
            tree_id=self.instance.tree_id,
            parent=self.instance
        )

    @property
    def is_empty(self):
        return not (self.children_departments or self.population() or self.vacancies())

    @property
    def old_state(self):
        if not hasattr(self, '_old_state'):
            if self.instance.id:
                self._old_state = Department.objects.get(id=self.instance.id)
            else:
                self._old_state = None
        return self._old_state

    def __getattr__(self, name):
        try:
            return super(DepartmentCtl, self).__getattr__(name)
        except AttributeError:
            if name in self.CTL_FIELDS:
                raise
            return getattr(self.instance, name)

    def __setattr__(self, name, value):

        is_department_attr = (
            name not in ['instance', '_author'] and
            hasattr(self.instance, name) and
            name not in self.CTL_FIELDS
        )

        if is_department_attr:
            setattr(self.instance, name, value)
        else:
            super(DepartmentCtl, self).__setattr__(name, value)

    def _perform_delayed_effects(self):
        """
        Запуск эффектов, которые меняют связанные сущности.
        """
        effects = []
        for effect in self._delayed_effects:
            if effect not in effects:
                effects.append(effect)
        for effect in effects:
            effect(self)
        self._delayed_effects = []

    @classmethod
    def create(cls, code, kind_id, author_user, oebs_structure_date: date, parent=None):
        if parent is None:
            parent_group = None
            url = code
            native_lang = 'ru'
            bg_color, fg_color = '#FFFFFF', '#000000'
        else:
            parent_group = Group.objects.get(department=parent)
            url = '{0}_{1}'.format(parent.url, code)
            native_lang = parent.native_lang
            bg_color, fg_color = parent.bg_color, parent.fg_color

        last_position = get_last_position(parent)

        now = datetime.now()
        base_params = dict(
            intranet_status=1,
            created_at=now,
            modified_at=now,
            url=url,
            code=code,
            position=last_position + 1,
            native_lang=native_lang,
        )

        try:
            department = Department.objects.create(
                from_staff_id=0,
                parent=parent,
                kind_id=kind_id,
                fg_color=fg_color,
                bg_color=bg_color,
                oebs_creation_date=oebs_structure_date,
                **base_params
            )
        except IntegrityError as e:
            raise DepartmentCtlError(
                str(e),
                code='db_integrity_error',
                params={'message': str(e.args)}
            )
        # TODO: через контроллер
        Group.objects.create(
            department=department,
            parent=parent_group,
            type=GROUP_TYPE_CHOICES.DEPARTMENT,
            **base_params
        )

        create_log(
            objects=[department],
            who=author_user,
            action='department_created',
            primary_key=department.pk
        )
        send_department_created(department)
        return cls(department=department, author_user=author_user)

    def save(self):
        if self._delayed_effects:
            self.old_state

        self.instance.modified_at = datetime.now()

        try:
            self.instance.save()

            if hasattr(self.instance, 'allowedheadcountoverdraft'):
                self.instance.allowedheadcountoverdraft.save()
        except InvalidMove as e:
            # надо будет ловить его контроллером заявки ещё до вьюхи, и
            # заполнить params айдишником экшена и райзить дальше чтобы вьюха отдала
            # фронту с айдишником экшена и пользователь знал какое действие вызвало ошибку
            raise DepartmentCtlError(
                str(e),
                code='mptt_invalid_move',
                params={'id': self.instance.id},
            )
        self._perform_delayed_effects()

        administration_objects = (
            DepartmentStaff.objects
            .filter(
                department=self.instance,
                role_id__in=[DepartmentRoles.CHIEF.value, DepartmentRoles.DEPUTY.value]
            )
        )
        log_objects = [self.instance] + list(administration_objects)
        create_log(
            objects=log_objects,
            who=self._author,  # user?
            action='department_updated',
            primary_key=self.id
        )
        return self

    def delete(self):
        """
        Удаление подразделения
        1. проверяет, что в подразделении не осталось людей и вакансий;
        2. рекурсивно удаляет все вложенные (пустые) подразделения;
        3. удаляет связанные объекты DepartmentStaff;
        4. выполняет "мягкое" удаление и отправляет сигнал department_deleted.
        """
        if self.has_population():
            raise DepartmentCtlError(
                'department is not empty',
                code='department_is_not_empty',
                params={
                    'dep_id': self.instance.id
                },
            )

        for child_dep in self.children_departments:
            child_dep_ctl = self.__class__(child_dep, self._author)
            child_dep_ctl.delete()

        related_administrations_filter = (
            Q(role_id=DepartmentRoles.CHIEF.value, staff=self.chief) |
            Q(role_id=DepartmentRoles.DEPUTY.value, staff__in=self.deputies)
        )

        (
            DepartmentStaff.objects
            .filter(department=self.instance)
            .filter(related_administrations_filter)
            .delete()
        )

        self.instance.intranet_status = 0
        self.instance.save(force_update=True)

        self.group.intranet_status = 0
        self.group.save(force_update=True)

        send_department_deleted(self.instance, self.instance.url)

    def has_population(self):
        if self.population().exists() or (
            waffle.switch_is_active('enable_vacancies_in_proposal')
            and self.vacancies().exists()
        ):
            return True

        return False
