class CtlField(object):
    def __init__(self, field_name, pre_effects=None,
                 post_effects=None, delayed_effects=None, validators=None):
        self.field_name = field_name
        self.pre_effects = pre_effects or []
        self.post_effects = post_effects or []
        self.delayed_effects = delayed_effects or []
        self.validators = validators or []

    def __get__(self, instance, owner):
        return getattr(instance.instance, self.field_name)

    def __set__(self, instance, value):
        if getattr(instance.instance, self.field_name) == value:
            return
        for validator in self.validators:
            validator(instance, value)  # should raise something
        for effect in self.pre_effects:
            effect(instance)
        setattr(instance.instance, self.field_name, value)
        for effect in self.post_effects:
            effect(instance)

        for effect in self.delayed_effects:
            instance._delayed_effects.append(effect)


class CtlCustomField(object):
    """Поле контроллера, позволяющее переопределить геттер и сеттер"""

    def __init__(self, fname, getter_func, setter_func,
                 cache_attr=None, lazy_update=True):

        self.cache_attr = cache_attr or '_{}__value'.format(fname)
        self.getter = getter_func
        self.setter = setter_func
        self.lazy_update = lazy_update

    def __get__(self, instance, owner):
        if instance is None:
            return self

        if not hasattr(instance, self.cache_attr):
            setattr(instance, self.cache_attr, self.getter(instance))
        return getattr(instance, self.cache_attr)

    def __set__(self, instance, value):
        setattr(instance, self.cache_attr, value)

        if self.lazy_update:
            instance._delayed_effects.append(self._save)
        else:
            self._save(instance)

    def _save(self, instance, as_delayed_effect=True):
        self.setter(instance, getattr(instance, self.cache_attr))
