from datetime import datetime

from django.contrib.auth.models import AnonymousUser

from staff.audit.factory import create_log
from staff.person.models import Staff

from staff.departments.models import HeadcountPosition


class HeadcountCtl:
    # STAFF-13923 пока не изменяем департамент. в будущем это будет правильнее делать из реестра
    IMMUTABLE_FIELDS = {'pk', 'id', 'department'}
    ALL_FIELDS = {f.name for f in HeadcountPosition._meta.fields}
    EDITABLE_FIELDS = ALL_FIELDS - IMMUTABLE_FIELDS

    def __init__(self, headcount: HeadcountPosition):
        self.instance: HeadcountPosition = headcount
        self.CTL_FIELDS = [
            f for f in self.ALL_FIELDS
            if f in HeadcountCtl.__dict__
        ]

    def __getattr__(self, name):
        try:
            return super().__getattr__(name)
        except AttributeError:
            if name in self.CTL_FIELDS:
                raise
            return getattr(self.instance, name)

    def __setattr__(self, name, value):

        is_headcount_attr = (
            name != 'instance'
            and hasattr(self.instance, name)
            and name not in self.CTL_FIELDS
        )

        if is_headcount_attr:
            setattr(self.instance, name, value)
        else:
            super().__setattr__(name, value)

    def save(self, author_user=None):
        self.modified_at = datetime.now()
        self.instance.save()

        if isinstance(author_user, Staff):
            author_user = author_user.user
        if not (author_user is None or isinstance(author_user, AnonymousUser)):
            create_log(
                objects=[self.instance],
                who=author_user,
                action='headcount_updated',
                primary_key=self.instance.code,  # принимает только int, ¯\_(ツ)_/¯
            )
        return self

    def update(self, data):
        for attr, value in data.items():
            if attr in self.EDITABLE_FIELDS:
                setattr(self, attr, value)
        return self
