import logging
from collections import defaultdict
from copy import deepcopy
from datetime import date, datetime, time
from enum import Enum
from itertools import chain
from typing import Any, Dict, Iterator, List, Set, Optional

import attr

from staff.budget_position.workflow_service import OEBSError
from staff.lib.db import atomic
from staff.lib.log import log_context
from staff.lib.models.departments_chain import get_departments_tree
from staff.person.models import Staff
from staff.proposal.hr_deadlines import split_person_actions, NearestDeadline

from staff.departments.controllers.actions.person_action import PersonAction
from staff.departments.controllers.actions.proposal_actions import ProposalActions
from staff.departments.controllers.department_data_ctl import DepartmentDataCtl
from staff.departments.controllers.exceptions import ProposalCtlError, BpConflict
from staff.departments.controllers.proposal_action import (
    Action,
    is_creating,
    is_moving,
    is_person_moving_to_created_department,
    PersonSection,
    DepartmentSection,
)
from staff.departments.controllers.workflow_registry_updater import WorkflowRegistryUpdater
from staff.departments.edit.proposal_mongo import (
    delete_mongo_object,
    from_mongo_id,
    get_mongo_object,
    get_mongo_objects,
    get_records_count,
    save_mongo_object,
    to_mongo_id,
    update_mongo_object,
    update_mongo_field,
    utc_to_local,
)
from staff.departments.models import (
    Department,
    DepartmentKind,
    HeadcountPosition,
    ProposalMetadata,
    Vacancy,
    DEADLINE_TYPE,
)


logger = logging.getLogger('departments.edit_proposal_ctl')

"""
    Одна заявка включает в себя цепочку изменяемых подразделений,
    сохраняется в монге и характеризуется уникальным _id.

    Невыполненные заявки не должны перескаться по изменяемым подразделениям или сотрудникам.
"""
DATE_FORMAT = '%Y-%m-%d'
TIME_FORMAT = '%H:%M:%S.%f'
DATETIME_FORMAT = '{date_format}T{time_format}'.format(date_format=DATE_FORMAT, time_format=TIME_FORMAT)


def delete_action_record(department_id):
    return {'id': department_id, 'fake_id': '', 'delete': True}


def time_now():
    return datetime.utcnow().isoformat()[:-3]


def collect_person_ids_from_proposal(
        proposal_object: Dict[str, Any],
        author=True,
        chiefs=True,
        deputies=True,
        from_tickets=False) -> Set[int]:
    """Возвращает id всех людей из заявки"""
    proposal_person_ids = []
    if author:
        proposal_person_ids.append(proposal_object['author'])

    for dep_action in proposal_object['actions']:
        if dep_action.get(DepartmentSection.ADMINISTRATION.value):
            if deputies:
                proposal_person_ids.extend(dep_action[DepartmentSection.ADMINISTRATION.value]['deputies'])
            chief_id = dep_action[DepartmentSection.ADMINISTRATION.value]['chief']
            if chiefs and chief_id:
                proposal_person_ids.append(chief_id)

    logins = [act['login'] for act in proposal_object['persons']['actions']]

    if from_tickets:
        logins.extend(proposal_object['tickets'].get('deleted_persons', {}).keys())
        logins.extend(proposal_object['tickets'].get('persons', {}).keys())

    if logins:
        proposal_person_ids.extend(
            Staff.objects
            .filter(login__in=logins)
            .values_list('id', flat=True)
        )

    return set(proposal_person_ids)


def collect_department_ids_from_dep_actions(
        dep_actions: List[Action],
        editing: bool,
        parents_of_creating: bool,
        new_parents_of_moving: bool,
) -> List[int]:
    department_ids: List[int] = []

    if editing:
        department_ids.extend(act['id'] for act in dep_actions)

    if parents_of_creating:
        department_ids.extend(
            act[DepartmentSection.HIERARCHY.value]['parent']
            for act in dep_actions
            if is_creating(act)
        )

    if new_parents_of_moving:
        department_ids.extend(act[DepartmentSection.HIERARCHY.value]['parent'] for act in dep_actions if is_moving(act))

    return [_id for _id in department_ids if _id]


def collect_department_ids_from_person_actions(
        person_actions: List[Action],
        actual_deps: bool = True,
        new_deps: bool = False,
) -> List[int]:

    department_ids: List[int] = []
    if actual_deps:
        logins = {act['login'] for act in person_actions}
        department_ids.extend(Staff.objects.filter(login__in=logins).values_list('department_id', flat=True))

    if new_deps:
        department_urls = {
            act[PersonSection.DEPARTMENT.value]['department']
            for act in person_actions
            if PersonSection.DEPARTMENT.value in act
        }
        department_ids.extend(Department.objects.filter(url__in=department_urls).values_list('id', flat=True))

    return list(set(department_ids))


def collect_department_ids_from_vacancy_actions(vacancy_actions: List[Action]) -> List[int]:
    dep_urls = [act[PersonSection.DEPARTMENT.value] for act in vacancy_actions]
    return list(
        Department.objects
        .filter(url__in=dep_urls)
        .values_list('id', flat=True)
    )


def collect_department_ids_from_headcount_actions(headcount_actions: List[Action]) -> List[int]:
    dep_urls = [act[PersonSection.DEPARTMENT.value] for act in headcount_actions]
    return list(
        Department.objects
        .filter(url__in=dep_urls)
        .values_list('id', flat=True)
    )


def collect_department_ids(
        proposal_object: Dict[str, Any],
        parents=True,  # родительские подразделения создаваемых
        new_parents=True,  # подразделения КУДА переносим
        person_deps=False,  # подразделения людей, изменяемых в заявке
        from_tickets=False,  # подразделения людей, на которых создавались тикеты
        vacancy_deps=True,  # подразделения вакансий, изменяемых в заявке
        headcount_deps=False,  # подразделения изменяемых хедкаунтов
) -> List[int]:
    """
    Собирает id существующих подразделений из заявки
    """
    from_dep_actions = collect_department_ids_from_dep_actions(
        proposal_object['actions'],
        editing=True,
        parents_of_creating=parents,
        new_parents_of_moving=new_parents,
    )

    from_person_actions = collect_department_ids_from_person_actions(
        proposal_object['persons']['actions'],
        actual_deps=True,
        new_deps=True,
    ) if person_deps else []

    person_tickets = proposal_object['tickets'].get('persons', {})
    deleted_person_tickets = proposal_object['tickets'].get('deleted_persons', {})
    deps_from_tickets = list(
        Staff.objects
        .filter(login__in=chain(person_tickets.keys(), deleted_person_tickets.keys()))
        .values_list('department_id', flat=True)
    ) if from_tickets else []

    from_vacancy_actions = collect_department_ids_from_vacancy_actions(
        proposal_object['vacancies']['actions']
    ) if vacancy_deps else []

    from_headcount_actions = collect_department_ids_from_headcount_actions(
        proposal_object['headcounts']['actions']
    ) if headcount_deps else []

    all_ids = set(chain(
        from_dep_actions,
        from_person_actions,
        from_vacancy_actions,
        from_headcount_actions,
        deps_from_tickets
    ))

    return list(all_ids)


def to_serializable(obj, datetime_format=DATETIME_FORMAT, date_format=DATE_FORMAT, time_format=TIME_FORMAT):
    """ф-я, заменяющая несериализуемые типы данных в сериализуемые"""
    if isinstance(obj, (list, tuple, set)):
        return [to_serializable(item) for item in obj]
    if isinstance(obj, dict):
        return {key: to_serializable(obj[key]) for key in obj}
    if isinstance(obj, datetime):
        return obj.strftime(datetime_format)
    if isinstance(obj, date):
        return obj.strftime(date_format)
    if isinstance(obj, time):
        return obj.strftime(time_format)
    return obj


class ApprovementStatus(Enum):
    WAIT_CREATE = 'wait_create'
    WAIT_DELETE = 'wait_delte'
    WAIT_RERUN = 'wait_rerun'
    OK = 'ok'


class ApprovementTicketType(Enum):
    RESTRUCTURISATION = 'restructurisation'
    PERSONAL = 'personal'
    HEADCOUNT = 'headcount'
    VACANCY = 'vacancy'


@attr.s(auto_attribs=True)
class ProposalApprovement:
    ticket_key: str
    ticket_type: str
    uuid: str = None  # set by OK
    staff_uid: str = None  # set by staff for determine approvements, created by staff
    status: str = ApprovementStatus.WAIT_CREATE.value

    def as_dict(self):
        return attr.asdict(self)

    @classmethod
    def from_dict(self, value):
        return self(**value)


class ProposalCtl:
    """
    Контроллер заявки.
     - Общается с монгой для создания/получения/изменения заявки
     - Накатывает изменения из заявки на реальную
       структуру подразделений и связанные модели
    """

    def __init__(self, proposal_id: str = None, author: Staff = None):
        # TODO: не путать автора заявки и автора изменений;
        #  продумать сохранение всех авторов изменений заявки.
        self.author = author

        self._login_to_person_ctl = {}
        self._vacancy_id_to_vacancy_ctl = {}
        self._url_to_dep = {}
        self._fake_id_to_dep: Dict[str, Department] = {}
        self._id_to_office = {}
        self._id_to_org = {}
        self._changed_from_previous_state = {}

        self.proposal_object = None
        self.proposal_id = proposal_id
        if self.proposal_id:
            self.load(proposal_id=self.proposal_id)

        self._workflow_registry_updater = WorkflowRegistryUpdater(self)

    @classmethod
    def filter(cls, spec, sort=None) -> Iterator['ProposalCtl']:
        """Генератор заявок, подходящих по условию spec"""
        for mongo_object in get_mongo_objects(spec, sort=sort, limit=9999):
            yield cls.create_from_dict(mongo_object)

    @staticmethod
    def get_proposals_count(spec=None):
        return get_records_count(spec)

    @property
    def splitted_from(self) -> str:
        return self.proposal_object.get('splitted_from')

    @splitted_from.setter
    def splitted_from(self, proposal_id):
        self.proposal_object['splitted_from'] = proposal_id

    @property
    def splitted_to(self) -> str:
        return self.proposal_object.get('splitted_to')

    @splitted_to.setter
    def splitted_to(self, proposal_id):
        self.proposal_object['splitted_to'] = proposal_id

    @property
    def department_ids(self):
        if not self.proposal_object:
            return []
        return [
            int(p['id'])
            for p in self.department_actions if p['id']
        ]

    @property
    def department_fake_ids(self):
        if not self.proposal_object:
            return []
        return [
            p['fake_id']
            for p in self.department_actions if p['fake_id']
        ]

    @property
    def department_actions(self):
        for it in self.proposal_object['actions']:
            yield it

    @department_actions.setter
    def department_actions(self, value):
        self.proposal_object['actions'] = value

    @property
    def person_actions(self) -> Iterator[Action]:
        for it in self.proposal_object['persons'].get('actions', []):
            yield it

    @person_actions.setter
    def person_actions(self, value):
        self.proposal_object['persons']['actions'] = value

    @property
    def person_action_objs(self) -> Iterator[PersonAction]:
        return map(PersonAction.from_dict, self.person_actions)

    @property
    def not_finished_person_actions(self) -> Iterator[Action]:
        for action in self.proposal_object['persons'].get('actions', []):
            if not action.get('finished'):
                yield action

    @property
    def apply_at(self) -> date:
        apply_at = self.proposal_object['apply_at']
        if isinstance(apply_at, datetime):
            return apply_at.date()
        return apply_at

    def get_person_actions_split(self):
        return split_person_actions(self.person_actions, date.today(), self.apply_at)

    @property
    def person_logins(self) -> Iterator[str]:
        return (act['login'] for act in self.person_actions)

    @property
    def vacancy_actions(self):
        for it in self.proposal_object['vacancies'].get('actions', []):
            yield it

    @vacancy_actions.setter
    def vacancy_actions(self, value):
        self.proposal_object['vacancies']['actions'] = value

    @property
    def not_finished_vacancy_actions(self):
        for action in self.proposal_object['vacancies'].get('actions', []):
            if not action.get('finished'):
                yield action

    @property
    def headcount_actions(self):
        for it in self.proposal_object['headcounts'].get('actions', []):
            yield it

    @headcount_actions.setter
    def headcount_actions(self, value):
        self.proposal_object['headcounts']['actions'] = value

    @property
    def not_finished_headcount_actions(self):
        for action in self.proposal_object['headcounts'].get('actions', []):
            if not action.get('finished'):
                yield action

    @property
    def all_actions(self):
        return chain(self.department_actions, self.person_actions, self.vacancy_actions, self.headcount_actions)

    def get_action(self, department_id):
        """
        Возвращает данные одного action
            в котором изменяется указанное подразделение
        Либо None если такого нет
        """
        for action in self.department_actions:
            if action['id'] == department_id:
                return action
        return None

    @property
    def dep_ticket(self):
        tickets = self.proposal_object['tickets']
        return tickets['department_ticket'] or tickets['department_linked_ticket'] or ''

    @property
    def restructurisation_ticket(self):
        tickets = self.proposal_object['tickets']
        return tickets.get('restructurisation', '')

    @property
    def value_stream_ticket(self):
        tickets = self.proposal_object['tickets']
        return tickets.get('value_stream', '')

    @property
    def headcount_ticket(self):
        tickets = self.proposal_object['tickets']
        return tickets.get('headcount', '')

    def get_all_actions(self) -> ProposalActions:
        return ProposalActions.from_proposal_obj(self.proposal_object)

    def get_approvements(self) -> List[ProposalApprovement]:
        return list(map(
            ProposalApprovement.from_dict,
            self.proposal_object.setdefault('approvements', []),
        ))

    def get_approvement_uuid(self, ticket_key) -> Optional[str]:
        return next(
            (
                it['uuid']
                for it in self.proposal_object['approvements']
                if it['ticket_key'] == ticket_key
            ),
            None,
        )

    def update_approvement(
        self,
        ticket_key: str,
        ticket_type: ApprovementTicketType,
        uuid: Optional[str] = None,
        status: ApprovementStatus = ApprovementStatus.WAIT_CREATE,
        remove_uuid: bool = False,
    ) -> str:
        approvement = next(
            (
                it for it in self.proposal_object.setdefault('approvements', [])
                if it['ticket_key'] == ticket_key
            ),
            None,
        )
        if not approvement:
            approvement = ProposalApprovement(
                ticket_key,
                ticket_type.value,
            ).as_dict()
            self.proposal_object['approvements'].append(approvement)
        if remove_uuid:
            approvement['uuid'] = None
            approvement['staff_uid'] = None
        if status == ApprovementStatus.WAIT_CREATE:
            approvement['staff_uid'] = time_now()
        if uuid is not None:
            approvement['uuid'] = uuid
        approvement['status'] = status.value

        update_mongo_field(
            self.proposal_id,
            'approvements',
            self.proposal_object['approvements'],
        )
        return approvement['staff_uid']

    def create_from_cleaned_data(self, proposal_params):
        """
        proposal_params: данные из cleaned_data формы заявки.
        Считаю, что они уже валидные
        """
        self.proposal_object = {
            'author': self.author.id,
            'apply_at': None,
            'apply_at_hr': None,
            'auto_apply': False,
            'description': '',
            'created_at': time_now(),
            'pushed_to_oebs': None,
            'splitted_from': None,  # proposal_id прошлой заявки при расщеплении
            'splitted_to': None,  # proposal_id следующей заявки при расщеплении
            'root_departments': [],  # урлы корневых подразделений
            'effective_date': None,  # поле для OEBS - дата применения структурных изменений с учетом порога
            'tickets': {
                'department_ticket': '',  # тикет, созданный по заявке
                'department_linked_ticket': '',  # тикет, привязанный к заявке
                'persons': {},  # {логин: ключ} тикеты по сотрудникам из вкладки persons
                'splitted_persons': {},  # тикеты по сотрудникам для расщеплении
                'restructurisation': '',  # тикет про реструктуризацию
                'deleted_persons': {},  # login:key тикеты по удалённым из заявки людям
                'deleted_restructurisation': '',  # удаленный тикет на реструктуризацию
                'headcount': '',  # тикет про headcount
                'deleted_headcount': '',
                'value_stream': '',
                'deleted_value_stream': '',
            },
            'actions': [],  # Подразделения
            'splitted_actions': [],  # Подразделения для расщепления
            'persons': {  # Сотрудники
                'actions': [],
                'splitted_actions': [],
            },
            'vacancies': {  # Вакансии (+ офферы)
                'actions': [],
                'splitted_actions': [],
            },
            'headcounts': {  # Headcounts
                'actions': [],
                'splitted_actions': [],
            },
            'approvements': [],  # List[ProposalApprovement.as_dict()]
        }

        self.update(proposal_params)
        self.proposal_object['tickets']['department_linked_ticket'] = (
            proposal_params.get('tickets', {}).get('department_linked_ticket')
        )
        self.set_departments_additional_data()
        self.proposal_object['tickets'].update(
            proposal_params.get('tickets', {})
        )

        return self

    def enable_auto_apply(self):
        self.proposal_object['auto_apply'] = True

    @property
    def is_auto_applied(self):
        return self.proposal_object.get('auto_apply', False)

    @classmethod
    def create_from_dict(cls, proposal_object):
        obj = cls()
        obj.fill_from_dict(proposal_object)
        return obj

    def get_ticket_by_person_action(self, action):
        return (
            self.proposal_object
            .get('tickets', {})
            .get('persons', {})
            .get(action['login'])
        )

    def fill_from_dict(self, proposal_object):
        """Наполняет объёкт по переданному дикту с данными"""
        proposal_object.setdefault('persons', {'actions': []})
        proposal_object.setdefault('vacancies', {'actions': []})
        proposal_object.setdefault('headcounts', {'actions': []})
        self.proposal_object = proposal_object
        self.proposal_id = from_mongo_id(self.proposal_object['_id'])

        if self.proposal_object['apply_at'] is not None:
            self.proposal_object['apply_at'] = self.apply_at

        departments = Department.objects.in_bulk(self.department_ids)
        for action in self.department_actions:
            if action['fake_id']:
                self._fake_id_to_dep[action['fake_id']] = departments.get(action['id'], None)

        self.author = Staff.objects.get(id=self.proposal_object['author'])

    def load(self, proposal_id):
        """Загружает объект из базы по переданному _id"""
        proposal_object = get_mongo_object(proposal_id)
        if not proposal_object:
            logger.error('Proposal with id %s does not exist', proposal_id)
            raise ProposalCtlError(
                'Proposal with id {} does not exist'.format(proposal_id),
                code='proposal_does_not_exist',
                params={
                    'proposal_id': proposal_id
                }
            )
        self.fill_from_dict(proposal_object)

    @atomic
    def save(self):
        if not self.proposal_object:
            raise ProposalCtlError('Nothing to save', 'nothing_to_save')

        if not self.is_deleted:
            self.set_departments_additional_data()

        if self.proposal_id:
            with log_context(proposal_id=self.proposal_id, expected_exceptions=[BpConflict, OEBSError]):
                if not self.is_deleted:
                    if not self.applied_at():
                        self.update_workflow_changes()

                    self.update_structure_change_date()
                self._changed_from_previous_state = {}
                update_mongo_object(self.proposal_id, self.proposal_object)
        else:
            try:
                self.proposal_id = save_mongo_object(self.proposal_object)
                with log_context(proposal_id=self.proposal_id, expected_exceptions=[BpConflict, OEBSError]):
                    ProposalMetadata.objects.get_or_create(
                        proposal_id=self.proposal_id,
                        defaults={'author': self.author},
                    )

                    self.update_workflow_changes()
                    self._changed_from_previous_state = {}
            except Exception:
                logging.warning('Error creating proposal', exc_info=True)
                delete_mongo_object(self.proposal_id)
                raise
        return self.proposal_id

    def _get_params_for_dep_action(self, action):
        """
         - Определить какие из табиков активны, остальные - игнорировать
         - Наполнить данные реальными объектами (Staff, Department, DepartmentKind)
         - Cделать плосче deputies
         - вернуть полученный дикт, который можно натягивать на DepartmentCtl
         - Определиться с parent
        """
        action_params = {'id': action['id']} if action.get('id') else {'fake_id': action['fake_id']}
        for key, val in action.items():
            if isinstance(val, dict) and DepartmentSection.has_value(key):
                action_params.update(val)
        if action.get('delete'):
            action_params['delete'] = True

        if action_params.get('chief'):
            action_params['chief'] = (
                Staff.objects.get(id=action_params['chief'])
            )
        if 'deputies' in action_params:
            action_params['deputies'] = (
                Staff.objects.filter(id__in=action_params['deputies'])
            )
        if DepartmentSection.HIERARCHY.value in action:
            parent = self._fake_id_to_dep.get(action_params.pop('fake_parent', None))
            action_params['parent'] = parent or Department.objects.filter(id=action_params.get('parent')).first()

        for fname in ('clubs', 'maillists'):
            if fname in action_params:
                action_params[fname] = [list(d.values())[0] for d in action_params[fname]]

        if action_params.get('kind'):
            action_params['kind'] = DepartmentKind.objects.get(id=action_params.get('kind'))

        return action_params

    def lock(self):
        self.proposal_object['locked'] = time_now()
        self.save()

    def unlock(self):
        self.proposal_object['locked'] = False
        self.save()

    def mark_pushed(self):
        meta = ProposalMetadata.objects.get(proposal_id=self.proposal_id)
        meta.pushed_to_oebs = time_now()
        meta.save()

    def _split_person_actions_for_new_proposal(self, logins: Set[str]):
        actions = []
        splitted_actions = []

        for action in self.person_actions:
            if action['login'] in logins:
                splitted_actions.append(action)
            else:
                actions.append(action)

        self.person_actions = actions
        self.proposal_object['persons']['splitted_actions'] = splitted_actions

        self.proposal_object['tickets']['splitted_persons'] = {
            action['login']: self.proposal_object['tickets']['persons'].pop(action['login'])
            for action in splitted_actions
            if action['login'] in self.proposal_object['tickets']['persons']
        }

    def _split_deleted_departments_for_new_proposal(self, department_ids: Set[int]):
        actions = []
        splitted_actions = []

        for action in self.department_actions:
            if action.get('id') in department_ids:
                splitted_actions.append(action)
            else:
                actions.append(action)

        self.department_actions = actions
        self.proposal_object['splitted_actions'] = splitted_actions

    def split_actions_for_new_proposal(self, splitting_data):
        self._split_person_actions_for_new_proposal(set(splitting_data.persons))
        self._split_deleted_departments_for_new_proposal(set(splitting_data.departments))

        assert len(list(self.all_actions)) > 0

        self.save()

    def rollback_actions_state(self, error_message=None):
        for action in self.all_actions:
            action['finished'] = False
            action.pop('_execution_order', None)

            if is_creating(action):
                action['id'] = None

            if is_person_moving_to_created_department(action):
                action[PersonSection.DEPARTMENT.value]['department'] = None

        splitted_department_actions = self.proposal_object.pop('splitted_actions', [])
        self.department_actions = list(self.department_actions) + splitted_department_actions

        splitted_person_actions = self.proposal_object['persons'].pop('splitted_actions', [])
        self.person_actions = list(self.person_actions) + splitted_person_actions

        splitted_person_tickets = self.proposal_object['tickets'].pop('splitted_persons', {})
        self.proposal_object['tickets']['persons'].update(splitted_person_tickets)

        if error_message:
            self.proposal_object['last_error'] = '{last}\n{time}: {message}'.format(
                last=self.proposal_object.get('last_error', ''),
                time=time_now(),
                message=error_message,
            )

    def update_workflow_changes(self):
        self._workflow_registry_updater.update_changes_for_proposal(
            self.get_person_actions_split(),
            list(self.vacancy_actions),
            list(self.headcount_actions),
            self.proposal_object['tickets'],
        )

    def _mark_person_actions_as_changed(self):
        self._changed_from_previous_state['person_actions_diff'] = {
            'updated': [it['login'] for it in self.person_actions],
            'created': [],
            'deleted': [],
        }

    @property
    def locked(self):
        return self.proposal_object.get('locked', False)

    def applied_at(self):
        metadata = (
            ProposalMetadata.objects
            .filter(proposal_id=self.proposal_id)
            .first()
        )
        return metadata and metadata.applied_at

    @property
    def is_deleted(self):
        return bool(
            ProposalMetadata.objects
            .filter(proposal_id=self.proposal_id)
            .values_list('deleted_at', flat=True)
            .first()
            or self.proposal_object.get('deleted_at')
        )

    def delete(self):
        with log_context(proposal_id=self.proposal_id):
            logger.info('Deleting proposal %s', self.proposal_id)
            delete_time = time_now()
            self._workflow_registry_updater.cancel_all_changes_for_proposal()
            ProposalMetadata.objects.filter(proposal_id=self.proposal_id).update(deleted_at=delete_time)
            update_mongo_object(self.proposal_id, self.proposal_object)

    def _update_proposal_objects(self, new_proposal_data):
        """
        Обновляет self.proposal_object, возвращает diff с id изменяемых экшенов
        если возвращаемый diff - пустой дикт - значит заявка не менялась
        """
        need_rewrite_changes = False
        for field_name in ('apply_at', 'apply_at_hr', 'description'):
            old_value = self.proposal_object.get(field_name)
            new_value = new_proposal_data.get(field_name)
            if old_value != new_value:
                self._changed_from_previous_state[field_name] = {
                    'old': old_value,
                    'new': new_value,
                }
                self.proposal_object[field_name] = new_value
                if field_name == 'apply_at':
                    need_rewrite_changes = True

        old_dep_actions = {
            act['id'] or act['fake_id']: act
            for act in self.department_actions
        }
        for dep_id in old_dep_actions:
            old_dep_actions[dep_id].pop('_execution_order', None)
        new_dep_actions = {
            act['id'] or act['fake_id']: act
            for act in new_proposal_data['actions']
        }
        added = set(new_dep_actions) - set(old_dep_actions)
        deleted = set(old_dep_actions) - set(new_dep_actions)

        updated = []
        for dep_id in set(new_dep_actions) & set(old_dep_actions):
            dep_action_old_state = old_dep_actions[dep_id].copy()
            dep_action_old_state.pop('__department_chain__', None)
            dep_action_new_state = new_dep_actions[dep_id]
            if dep_action_old_state != dep_action_new_state:
                updated.append(dep_id)

        if any([added, deleted, updated]):
            self._changed_from_previous_state['actions'] = {
                'added': added,
                'deleted': deleted,
                'updated': updated,
            }
        self.department_actions = new_proposal_data['actions']
        self.proposal_object['_prev_actions_state'] = {
            str(dep_id): dep_data
            for dep_id, dep_data in old_dep_actions.items()
        }

        new_person_actions = new_proposal_data.get('persons', {'actions': []})['actions']
        self._changed_from_previous_state['person_actions_diff'] = self._update_person_actions(new_person_actions)
        if need_rewrite_changes:
            self._mark_person_actions_as_changed()

        new_vacancies_actions = new_proposal_data.get('vacancies', {'actions': []})['actions']
        self._changed_from_previous_state['vacancy_actions_diff'] = self._update_vacancy_actions(new_vacancies_actions)

        new_headcount_actions = new_proposal_data.get('headcounts', {'actions': []})['actions']
        self._changed_from_previous_state['headcount_actions_diff'] = self._update_headcount_actions(
            new_headcount_actions,
        )

    def _update_person_actions(self, new_person_actions):
        result = {
            'created': [],
            'deleted': [],
            'updated': [],
        }
        id_to_old_and_new = defaultdict(lambda: [None, None])
        new_state = []
        for act in self.person_actions:
            id_to_old_and_new[act['action_id']][0] = act
        for act in new_person_actions:
            id_to_old_and_new[act['action_id']][1] = act
        for old_act, new_act in id_to_old_and_new.values():
            if old_act is None:
                new_state.append(new_act)
                result['created'].append(new_act['login'])
            elif new_act is None:
                result['deleted'].append(old_act['login'])
            elif any(old_act.get(k) != v for k, v in new_act.items()):
                result['updated'].append(new_act['login'])
                new_state.append(deepcopy(new_act))
            else:
                new_state.append(old_act)
        if any(result.values()):
            self.person_actions = new_state
        return result

    def _update_vacancy_actions(self, new_vacancy_actions):
        result = {
            'created': [],
            'deleted': [],
            'updated': [],
        }
        id_to_old_and_new = defaultdict(lambda: [None, None])
        new_state = []
        for act in self.vacancy_actions:
            id_to_old_and_new[act['action_id']][0] = act
        for act in new_vacancy_actions:
            id_to_old_and_new[act['action_id']][1] = act
        for old_act, new_act in id_to_old_and_new.values():
            if old_act is None:
                new_state.append(new_act)
                result['created'].append(new_act['vacancy_id'])
            elif new_act is None:
                result['deleted'].append(old_act['vacancy_id'])
            elif any(old_act.get(k) != v for k, v in new_act.items()):
                result['updated'].append(new_act['vacancy_id'])
                new_state.append(deepcopy(new_act))
            else:
                new_state.append(old_act)
        if any(result.values()):
            self.vacancy_actions = new_state
        return result

    def _update_headcount_actions(self, new_headcount_actions):  # copypaste?
        result = {
            'created': [],
            'deleted': [],
            'updated': [],
        }
        id_to_old_and_new = defaultdict(lambda: [None, None])
        new_state = []
        for act in self.headcount_actions:
            id_to_old_and_new[act['action_id']][0] = act
        for act in new_headcount_actions:
            id_to_old_and_new[act['action_id']][1] = act
        for old_act, new_act in id_to_old_and_new.values():
            if old_act is None:
                new_state.append(new_act)
                result['created'].append(new_act['headcount_code'])
            elif new_act is None:
                result['deleted'].append(old_act['headcount_code'])
            elif any(old_act.get(k) != v for k, v in new_act.items()):
                result['updated'].append(new_act['headcount_code'])
                new_state.append(deepcopy(new_act))
            else:
                new_state.append(old_act)
        if any(result.values()):
            self.headcount_actions = new_state
        return result

    def update(self, proposal_params):
        self.proposal_object['updated_at'] = time_now()
        self._update_proposal_objects(proposal_params)
        return to_serializable(self._changed_from_previous_state)

    def update_tickets(
        self,
        person_tickets: Dict[str, str] = None,
        deleted_person_tickets: Dict[str, str] = None,
        restructurisation_ticket: str = None,
        headcount_ticket: str = None,
        value_steam_ticket: str = None,
    ):
        person_tickets = person_tickets or {}
        deleted_person_tickets = deleted_person_tickets or {}

        conflict_logins = list(set(person_tickets) & set(deleted_person_tickets))
        if conflict_logins:
            raise ValueError(f'Logins {conflict_logins} are both deleted and not. Proposal: {self.proposal_id}')

        for login, ticket_key in person_tickets.items():
            self.proposal_object['tickets']['persons'][login] = ticket_key
            self.proposal_object['tickets']['deleted_persons'].pop(login, '')

        for login, ticket_key in deleted_person_tickets.items():
            self.proposal_object['tickets']['deleted_persons'][login] = ticket_key
            self.proposal_object['tickets']['persons'].pop(login, '')

        if restructurisation_ticket is not None:
            self.proposal_object['tickets']['restructurisation'] = restructurisation_ticket

        if headcount_ticket is not None:
            self.proposal_object['tickets']['headcount'] = headcount_ticket

        if value_steam_ticket is not None:
            self.proposal_object['tickets']['value_stream'] = value_steam_ticket

        # таска может приходить апдейтить тикеты уже после удаления заявки
        # в таком случае реестр не дергаем, там уже все отменено
        if self.is_deleted:
            logger.info('Will not update registry, as proposal already removed')
            return

        self.update_workflow_changes()

    def get_updated_actions(self):
        actions = {}
        if not self.proposal_object:
            return actions

        for department_action in self.department_actions:
            if department_action['fake_id']:
                actions[department_action['fake_id']] = [
                    section.value
                    for section in DepartmentSection
                    if section.value in department_action
                ]
            if department_action['id']:
                actions[int(department_action['id'])] = [
                    section.value
                    for section in DepartmentSection
                    if section.value in department_action
                ]
        return actions

    def as_form_data(self):
        if not self.proposal_object:
            return {}

        proposal = deepcopy(self.proposal_object)
        proposal.pop('old_state', None)

        old_data = DepartmentDataCtl(self.department_ids).as_actions()
        for department_action in proposal['actions']:
            if department_action['fake_id'] and not department_action['id']:
                continue

            if not department_action['id'] in old_data:
                # can happen when we see data from mongo during proposal execution
                # but can't find department in db, as proposal transaction still not finished
                continue

            for section in DepartmentSection:
                department_action[section.value] = department_action.get(
                    section.value, old_data[department_action['id']][section.value]
                )
            # костыль чтобы показывать код на форме если меняли position
            if 'technical' in department_action:
                department_action['technical']['code'] = (
                    old_data.get(department_action['id'], {})['technical']['code']
                )
        if '_id' in proposal:
            applied_at, applied_by = (
                ProposalMetadata.objects
                .values_list('applied_at', 'applied_by__login')
                .get(proposal_id=from_mongo_id(proposal['_id']))
            )

            proposal['applied_at'] = applied_at
            proposal['applied_by'] = applied_by

        return proposal

    def get_creation_date(self):
        if self.proposal_object and self.proposal_object['created_at']:
            return datetime.strptime(self.proposal_object['created_at'], DATETIME_FORMAT).date()
        return None

    def get_conflicts(self, exclude_ids=None):
        """
        Проверить нету ли в заявке подразделений,
        которые есть в невыполненных заявках в монге
        exclude_ids - список id заявок, исключаемых из проверки
        """
        exclude_ids = exclude_ids or []
        unfinished_proposal_ids = list(
            ProposalMetadata.objects
            .filter(applied_at=None, deleted_at=None)
            .exclude(proposal_id__in=exclude_ids)
            .values_list('proposal_id', flat=True)
        )
        unfinished_proposals = get_mongo_objects(
            spec={
                '_id': {'$in': [to_mongo_id(_id) for _id in unfinished_proposal_ids]},
                'actions.id': {'$in': self.department_ids}
            },
        )
        unfinished_proposals = (
            from_mongo_id(p['_id'])
            for p in unfinished_proposals
        )
        return [_id for _id in unfinished_proposals if _id not in exclude_ids]

    def _set_root_departments(self) -> None:
        """Кладёт в тело заявки по ключу `root_departments` урлы корневых подразделений,
        в ветках которых происходят департаментные изменения в рамках этой заявки"""
        involved_departments = []
        for dep_act in self.department_actions:
            if dep_act['id']:
                involved_departments.append(dep_act['id'])
            hierarchy_action = dep_act.get(DepartmentSection.HIERARCHY.value)
            if hierarchy_action and hierarchy_action.get('parent'):
                involved_departments.append(hierarchy_action['parent'])

        department_chains = get_departments_tree(
            involved_departments,
            fields=['url'],
        )
        root_departments = set(
            [
                dep_chain[0]['url']
                for dep_chain in department_chains.values()
            ]
        )
        self.proposal_object['root_departments'] = list(root_departments)

    def _set_department_paths(self) -> None:
        """
        Добавляет по ключу `__department_chain__`
        цепочки подразделений во все экшены.
        """
        # persons, vacancies, headcounts

        involved_departments = []

        logins = (act['login'] for act in self.person_actions)
        login_to_department_id = dict(
            Staff.objects
            .filter(login__in=logins)
            .values_list('login', 'department_id')
        )
        involved_departments.extend(login_to_department_id.values())

        self.vacancy_actions = list(self.vacancy_actions)
        vacancy_ids = (act['vacancy_id'] for act in self.vacancy_actions)
        vacancy_id_to_department_id = dict(
            Vacancy.objects
            .filter(id__in=vacancy_ids)
            .values_list('id', 'department_id')
        )
        involved_departments.extend(vacancy_id_to_department_id.values())

        self.headcount_actions = list(self.headcount_actions)
        headcounts_codes = (act['headcount_code'] for act in self.headcount_actions)
        headcount_code_to_department_id = dict(
            HeadcountPosition.objects
            .filter(code__in=headcounts_codes)
            .values_list('code', 'department_id')
        )
        involved_departments.extend(headcount_code_to_department_id.values())

        involved_departments.extend(collect_department_ids(self.proposal_object))
        department_chains = get_departments_tree(involved_departments, fields=['url'])
        department_chains = {
            pk: [c['url'] for c in chain]
            for pk, chain in department_chains.items()
        }

        for action in self.person_actions:
            dep_id = login_to_department_id[action['login']]
            action['__department_chain__'] = department_chains[dep_id]

        for action in self.vacancy_actions:
            dep_id = vacancy_id_to_department_id[action['vacancy_id']]
            action['__department_chain__'] = department_chains[dep_id]

        for action in self.headcount_actions:
            dep_id = headcount_code_to_department_id[action['headcount_code']]
            action['__department_chain__'] = department_chains[dep_id]
        # departments

        actions_creating = [act for act in self.department_actions if is_creating(act)]
        id_to_parent = {
            act['fake_id']: act[DepartmentSection.HIERARCHY.value]['parent']
            for act in actions_creating
            if act[DepartmentSection.HIERARCHY.value]['parent']
        }

        id_to_fake_parent = {
            act['fake_id']: act[DepartmentSection.HIERARCHY.value]['fake_parent']
            for act in actions_creating
            if act[DepartmentSection.HIERARCHY.value]['fake_parent']
        }

        def find_parent(fake_id, fake_parent):
            if fake_parent in id_to_parent:
                id_to_parent[fake_id] = id_to_parent[fake_parent]
                return id_to_parent[fake_parent]
            real_parent = find_parent(fake_parent, id_to_fake_parent[fake_parent])
            id_to_parent[fake_id] = real_parent
            id_to_fake_parent.pop(fake_id, None)
            return real_parent

        while id_to_fake_parent:
            find_parent(*id_to_fake_parent.popitem())

        assert len(id_to_parent) == len(actions_creating)

        for action in self.department_actions:
            if action['fake_id'] in id_to_parent:
                action['__department_chain__'] = department_chains[id_to_parent[action['fake_id']]]
            elif action['id']:
                action['__department_chain__'] = department_chains[action['id']]

    def set_departments_additional_data(self) -> None:
        """
        Добавляет в тело заявки дополнительные данные
        для разметки сущностей по подразделениям.
        """
        if not self.proposal_object:
            return

        self._set_root_departments()
        self._set_department_paths()

    def get_meta(self):
        """
        Собирает метаинформацию о всех подразделениях
        + инфа о том какие экшены изменялись в рамках заявки
        + инфа о состоянии заявки
        """
        meta = DepartmentDataCtl(self.department_ids).as_meta()
        actions = self.get_updated_actions()
        for id in chain(self.department_ids, self.department_fake_ids):
            if id in meta:
                meta[id]['actions'] = actions[id]
            else:
                meta[id] = {'actions': actions[id]}
        meta['locked'] = self.locked
        meta['applied_at'] = self.applied_at() or False
        meta['actions'] = [
            {'finished': act.get('finished', False)}
            for act in self.department_actions
        ]
        return meta

    def update_structure_change_date(self):
        deadline = NearestDeadline().find_for_deadline_type(datetime.today(), DEADLINE_TYPE.STRUCTURE_CHANGE)
        assert isinstance(deadline.month, date)
        self.proposal_object['effective_date'] = deadline.month

    @property
    def effective_date(self) -> date:
        result = self.proposal_object['effective_date']
        if isinstance(result, datetime):
            return result.date()
        return result


def _format_action_for_oebs(action, department_data, login_by_id):

    categories = {
        'technical': '100',
        'nontechnical': '200',
        'other': '300',  # Иное. Пока этого значения нет в DIS
    }

    if is_creating(action):
        action_type = 'create'
    elif DepartmentSection.NAME.value in action:
        action_type = 'update'
    else:
        action_type = ''

    oebs_action_data = {
        'global_org_id': action['id'],
        'delete': action['delete'],
        'execution_order': action.get('_execution_order', 0),
        'name': {
            'type': action_type,
            'kind': str(department_data['kind__slug']),
            'name': department_data['name'],
            'name_en': department_data['name_en'],
        },
        'extra': {
            'category': categories.get(department_data['category'], categories['other']),
        },
    }

    oebs_action_data['name']['hr_type'] = (
        action[DepartmentSection.NAME.value].get('hr_type', True) if DepartmentSection.NAME.value in action else True
    )

    oebs_action_data['name']['is_correction'] = (
        action[DepartmentSection.NAME.value].get('is_correction', False)
        if DepartmentSection.NAME.value in action
        else False
    )

    if DepartmentSection.HIERARCHY.value in action:
        oebs_action_data['hierarchy'] = {
            'parent': action[DepartmentSection.HIERARCHY.value]['parent']
        }
    if DepartmentSection.ADMINISTRATION.value in action:
        oebs_action_data['administration'] = {
            'chief': login_by_id.get(action[DepartmentSection.ADMINISTRATION.value]['chief']),
            'deputies': [
                login_by_id.get(person_id)
                for person_id in action[DepartmentSection.ADMINISTRATION.value]['deputies']
            ],
        }

    return oebs_action_data


def proposal_oebs_format(proposal_object):
    """Возвращает данные по заявке в удобном для ОЕБС виде"""

    proposal_id = proposal_object['_id']
    applied_at, applied_by = (
        ProposalMetadata.objects
        .values_list('applied_at', 'applied_by__login')
        .get(proposal_id=proposal_id)
    )

    departments_data = {
        d['id']: d
        for d in Department.objects
        .filter(id__in=collect_department_ids(proposal_object))
        .values(
            'id',
            'kind__slug',
            'name', 'name_en', 'short_name', 'short_name_en',
            'category',
        )
    }

    persons_logins = dict(
        Staff.objects
        .filter(id__in=collect_person_ids_from_proposal(proposal_object))
        .values_list('id', 'login')
    )

    ticket_key = (
        proposal_object['tickets']['department_ticket']
        or proposal_object['tickets']['department_linked_ticket']
        or proposal_object['tickets'].get('restructurisation', '')
    )

    fake2id = {
        a['fake_id']: a['id']
        for a in proposal_object['actions'] if is_creating(a)
    }
    for action in proposal_object['actions']:
        hierarchy_action = action.get(DepartmentSection.HIERARCHY.value)
        if hierarchy_action and hierarchy_action['fake_parent']:
            hierarchy_action['parent'] = fake2id.get(hierarchy_action['fake_parent'])

    return {
        'ticket': ticket_key,
        'description': proposal_object['description'],
        'applied_by': applied_by,
        'apply_at': (
            proposal_object.get('apply_at')
            .strftime('%Y-%m-%dT%H:%M:%S.%f')[:-3]
        ),
        'created_by': applied_by,  # Удалим когда OEBS с этого поля слезет
        'created_at': (            # Тоже удалим
            proposal_object.get('apply_at')
            .strftime('%Y-%m-%dT%H:%M:%S.%f')[:-3]
        ),
        'execution_date': utc_to_local(applied_at),
        'actions': [
            _format_action_for_oebs(action, departments_data.get(action['id']), persons_logins)
            for action in proposal_object['actions']
        ],
        'effective_date': proposal_object.get('effective_date'),
    }
