from enum import Enum
from itertools import count
from typing import Any, Dict, Iterable, Iterator, Set

from staff.lib.utils.ordered_choices import OrderedChoices

from staff.departments.controllers.exceptions import ProposalCtlError
from staff.departments.models import Department


class SectionsEnum(Enum):
    @classmethod
    def has_value(cls, value) -> bool:
        return value in cls._value2member_map_

    @classmethod
    def as_ordered_choices(cls) -> OrderedChoices:
        return OrderedChoices(*[
            (item.value, item.value)
            for item in cls
        ])


class DepartmentSection(SectionsEnum):
    NAME = 'name'
    HIERARCHY = 'hierarchy'
    ADMINISTRATION = 'administration'
    TECHNICAL = 'technical'


class PersonSection(SectionsEnum):
    SALARY = 'salary'
    POSITION = 'position'
    GRADE = 'grade'
    DEPARTMENT = 'department'
    OFFICE = 'office'
    ORGANIZATION = 'organization'
    VALUE_STREAM = 'value_stream'
    GEOGRAPHY = 'geography'


class PersonSectionsSet:
    @classmethod
    def from_person_section(cls, *args: PersonSection):
        return PersonSectionsSet(*[section.value for section in args])

    def __init__(self, *args: str):
        self._sections_set = {section for section in args} & {section.value for section in PersonSection}

    def has_any_section_from(self, other: 'PersonSectionsSet'):
        return bool(self._sections_set & other._sections_set)

    def only_section_from(self, other: 'PersonSectionsSet', is_the: PersonSection):
        return self._sections_set & other._sections_set == {is_the.value}

    def has_only_section(self, section: PersonSection):
        return self._sections_set == {section.value}


PERSON_ACTION_SECTIONS = PersonSection.as_ordered_choices()
DEPARTMENT_ACTION_SECTIONS = DepartmentSection.as_ordered_choices()

Action = Dict[str, Any]


def is_deleting(act: Action) -> bool:
    return act['delete']


def is_creating(act: Action) -> bool:
    return bool(act.get('fake_id'))


def is_moving(act: Action) -> bool:
    return DepartmentSection.HIERARCHY.value in act and not is_creating(act)


def is_structure_modification(act: Action) -> bool:
    return is_creating(act) or is_moving(act) or is_deleting(act)


def is_data_modification(act: Action) -> bool:
    return not is_structure_modification(act)


def is_executable_person_action(act: Action) -> bool:
    if not is_person_action(act):
        return False

    return any(type_ in act['sections'] for type_ in (
        PERSON_ACTION_SECTIONS.department,
        PERSON_ACTION_SECTIONS.office,
        PERSON_ACTION_SECTIONS.organization,
        PERSON_ACTION_SECTIONS.position,
    ))


def is_person_action(act: Action) -> bool:
    return 'login' in act


def is_person_moving(act: Action) -> bool:
    return is_person_action(act) and PersonSection.DEPARTMENT.value in act


def is_person_moving_to_created_department(act: Action) -> bool:
    return is_person_moving(act) and act[PersonSection.DEPARTMENT.value].get('fake_department')


def is_vacancy_action(act: Action) -> bool:
    return 'vacancy_id' in act


def is_headcount_action(act: Action) -> bool:
    return 'headcount_code' in act


def is_dep_action(act: Action) -> bool:
    return not is_person_action(act) and not is_vacancy_action(act) and not is_headcount_action(act)


def ordered_actions(actions: Iterable[Action],
                    with_not_executable_actions: bool = False) -> Iterator[Action]:
    """
    with_not_executable_actions - если True - отдаём вместе с экшенами, по которым нам ничего не надо делать (изм. зп).
    Выдаём экшены в таком порядке:
    1.  все экшены подразделений, помимо удалений, выполняются сверху вниз по деревьям заявки
    2.  изменяемые сотрудники
    3.  изменяемые вакансии
    4.  изменяемые бюджетные позиции
    5.  удаляемые подразделения, отсортированные по level, чтобы удалять снизу вверх
    """
    actions = list(actions)
    departments = Department.objects.in_bulk([act['id'] for act in actions if is_dep_action(act) and act['id']])
    department_actions: Dict[str, Action] = {}
    department_delete_actions, person_actions, vacancy_actions, headcount_actions = [], [], [], []

    for action in actions:
        if is_dep_action(action):
            if is_deleting(action):
                department_delete_actions.append(action)
            else:
                action_id = action['fake_id'] or departments[action['id']].url
                action.pop('_execution_order', None)
                department_actions[action_id] = action
        elif is_executable_person_action(action) or (with_not_executable_actions and is_person_action(action)):
            person_actions.append(action)
        elif is_vacancy_action(action):
            vacancy_actions.append(action)
        elif is_headcount_action(action):
            headcount_actions.append(action)

    order = count()

    def with_order(act):
        act['_execution_order'] = next(order)
        return act

    visited_actions: Set[str] = set()

    def process_department_action(action_id: str, action: Action):
        visited_actions.add(action_id)
        hierarchy_action = action.get(DepartmentSection.HIERARCHY.value)
        parent_id = hierarchy_action and (hierarchy_action['parent'] or hierarchy_action['fake_parent'])
        if parent_id and parent_id in department_actions:
            parent_action = department_actions[parent_id]
            if parent_id not in visited_actions:
                yield from process_department_action(parent_id, parent_action)
            elif '_execution_order' not in parent_action:
                raise ProposalCtlError('Cycle', code='cyclic_relations', params={'dep_ids': [action_id, parent_id]})
        yield with_order(action)

    for action_id, action in department_actions.items():
        if '_execution_order' not in action:
            yield from process_department_action(action_id, action)

    for action in person_actions:
        yield with_order(action)

    for action in vacancy_actions:
        yield with_order(action)

    for action in headcount_actions:
        yield with_order(action)

    for action in sorted(department_delete_actions, key=lambda act: departments[act['id']].level, reverse=True):
        yield with_order(action)
