from contextlib import contextmanager
from datetime import date
import logging
from typing import Dict


from staff.departments.controllers import department
from staff.departments.controllers.department_data_ctl import DepartmentDataCtl
from staff.departments.controllers.exceptions import ProposalCtlError
from staff.departments.controllers.headcount_ctl import HeadcountCtl
from staff.departments.controllers.intern_transfer_detector import InternTransferDetector
from staff.departments.controllers.proposal import ProposalCtl, time_now
from staff.departments.controllers.proposal_action import (
    is_executable_person_action,
    ordered_actions,
    is_dep_action,
    is_person_action,
    is_vacancy_action,
    is_headcount_action,
    DepartmentSection,
    PersonSection,
    Action,
    is_moving,
)
from staff.departments.controllers.vacancy import VacancyCtl
from staff.departments.controllers.workflow_registry_updater import WorkflowRegistryUpdater
from staff.departments.models import Department, Vacancy, DepartmentKind, ProposalMetadata, HeadcountPosition
from staff.lib.log import log_context
from staff.map.models import Office
from staff.person.controllers import PersonCtl
from staff.person.effects import actualize_affiliation
from staff.person.models import Staff, Organization


logger = logging.getLogger('proposal')


class ProposalExecution:
    def __init__(self, proposal_ctl: ProposalCtl, intern_transfer_detector: InternTransferDetector = None) -> None:
        assert proposal_ctl
        self.intern_transfer_detector = intern_transfer_detector or InternTransferDetector()
        self._proposal_ctl = proposal_ctl
        self._login_to_person_ctl = {}
        self._url_to_dep = {}
        self._id_to_office = {}
        self._id_to_org = {}
        self._vacancy_id_to_vacancy_ctl = {}
        self._headcount_code_to_headcount_ctl = {}
        self._fake_id_to_dep: Dict[str, Department] = {}

    @property
    def proposal_id(self):
        return self._proposal_ctl.proposal_id

    @contextmanager
    def lock_proposal(self):
        self._proposal_ctl.lock()
        try:
            yield
        except Exception:
            logger.exception("Error while proposal %s was locked", self.proposal_id)
            raise
        finally:
            self._proposal_ctl.unlock()

    def execute(self, execution_author_login=None):
        with log_context(proposal_id=self.proposal_id):
            if not self._proposal_ctl.proposal_object:
                raise ProposalCtlError('Nothing to save', 'nothing_to_save')

            logger.info('Executing proposal %s', self.proposal_id)

            self._save_departments_old_state()
            self._fetch_for_person_actions()
            self._fetch_for_vacancy_actions()
            self._fetch_for_headcount_actions()
            self._fill_departments_fake_ids()

            with self.lock_proposal():
                if date.today() > self._proposal_ctl.apply_at:
                    self._proposal_ctl.update_workflow_changes()

                for cur_action in ordered_actions(self._proposal_ctl.all_actions):
                    logger.info('Executing next action for %s', self.proposal_id)
                    if is_dep_action(cur_action):
                        self._execute_dep_action(cur_action)
                    elif is_person_action(cur_action):
                        self._execute_person_department_action(cur_action)
                        self._execute_person_action(cur_action)
                    elif is_vacancy_action(cur_action):
                        self._execute_vacancy_action(cur_action)
                    elif is_headcount_action(cur_action):
                        self._execute_headcount_action(cur_action)

                    cur_action['finished'] = time_now()
                    self._save()

                self.create_proposal_for_splitted_actions()

                # отчитываемся о проведённых изменениях
                self._proposal_ctl.proposal_object['last_error'] = None
                self._set_metadata_applied(execution_author_login)

                self._confirm_workflows_on_execute()
                self.actualize_affiliations()

    def _save(self):
        # так же будут отменены и заново созданы неисполненные записи в реестре
        self._proposal_ctl.save()

    def _fill_departments_fake_ids(self):
        departments = Department.objects.in_bulk(self._proposal_ctl.department_ids)
        for action in self._proposal_ctl.department_actions:
            if action['fake_id']:
                self._fake_id_to_dep[action['fake_id']] = departments.get(action['id'], None)

    def _save_departments_old_state(self):
        dep_ids = self._proposal_ctl.department_ids
        if not dep_ids:
            return
        dep_data_ctl = DepartmentDataCtl(dep_ids)

        self._proposal_ctl.proposal_object['old_state'] = {
            str(dep_id): dep_data_ctl.as_form_data(dep_id)
            for dep_id in dep_ids
        }

    def _fetch_for_person_actions(self):
        """
        Need to fetch person models and department ids for urls.
        """
        executable = [
            act for act in self._proposal_ctl.person_actions
            if is_executable_person_action(act)
        ]
        if not executable:
            return
        logins = []
        dep_urls = []
        org_ids = []
        office_ids = []
        for act in executable:
            logins.append(act['login'])
            url = act.get(PersonSection.DEPARTMENT.value, {}).get('department')
            if url:
                dep_urls.append(url)
            office_id = act.get(PersonSection.OFFICE.value, {}).get('office')
            if office_id:
                office_ids.append(office_id)
            org_id = act.get(PersonSection.ORGANIZATION.value, {}).get('organization')
            if org_id:
                org_ids.append(org_id)

        self._login_to_person_ctl.update(
            (person.login, PersonCtl(person))
            for person in Staff.objects.filter(login__in=logins)
        )

        if dep_urls:
            self._url_to_dep.update(
                (dep.url, dep)
                for dep in Department.objects.filter(url__in=dep_urls)
            )

        if office_ids:
            self._id_to_office.update(
                (office.id, office)
                for office in Office.objects.filter(id__in=office_ids)
            )

        if org_ids:
            self._id_to_org.update(
                (org.id, org)
                for org in Organization.objects.filter(id__in=org_ids)
            )

    def _fetch_for_vacancy_actions(self):
        """
        Need to fetch vacancy models and department ids for urls.
        """
        vacancy_ids = []
        dep_urls = []
        for act in self._proposal_ctl.vacancy_actions:
            vacancy_ids.append(act['vacancy_id'])
            url = act.get('department')
            if url:
                dep_urls.append(url)

        self._vacancy_id_to_vacancy_ctl.update(
            (vacancy.id, VacancyCtl(vacancy))
            for vacancy in Vacancy.objects.filter(id__in=vacancy_ids)
        )
        if dep_urls:
            self._url_to_dep.update(
                (dep.url, dep)
                for dep in Department.objects.filter(url__in=dep_urls)
            )

    def _fetch_for_headcount_actions(self):
        """
        Need to fetch headcounts models and department ids for urls.
        """
        headcounts_codes = []
        dep_urls = []
        for act in self._proposal_ctl.headcount_actions:
            headcounts_codes.append(act['headcount_code'])
            url = act.get('department')
            if url:
                dep_urls.append(url)

        self._headcount_code_to_headcount_ctl.update(
            (headcount.code, HeadcountCtl(headcount))
            for headcount in HeadcountPosition.objects.filter(code__in=headcounts_codes)
        )
        if dep_urls:
            self._url_to_dep.update(
                (dep.url, dep)
                for dep in Department.objects.filter(url__in=dep_urls)
            )

    def _execute_dep_action(self, action):
        action_params = self._get_params_for_dep_action(action)
        cur_id = action_params.pop('id', None)
        try:
            if cur_id:
                logger.info('Proposal %s. Running edit department %s', self.proposal_id, cur_id)
                self.edit_department(cur_id, action_params)
            else:
                logger.info('Proposal %s. Running create department %s', self.proposal_id, action_params.get('fake_id'))
                self.create_department(action_params)
        except Exception as e:
            action_num = action['_execution_order']
            # протестировать правильно ли ошибку прикладываю
            e.params = {'actions[{}]'.format(action_num): e.params}
            error_message = '#act{num} [{time}]: {msg}. params: {params}'.format(
                num=action_num,
                time=time_now(),
                msg=str(e),
                params=e.params,
            )
            logger.warning(
                'Proposal %s. exception while executing action %s. message: %s',
                self.proposal_id,
                action_num,
                error_message,
            )
            raise

        if not cur_id:
            cur_id = action_params.get('fake_id')
            action['id'] = self._fake_id_to_dep.get(cur_id).id

    def _get_params_for_dep_action(self, action):
        """
         - Определить какие из табиков активны, остальные - игнорировать
         - Наполнить данные реальными объектами (Staff, Department, DepartmentKind)
         - Cделать плосче deputies
         - вернуть полученный дикт, который можно натягивать на DepartmentCtl
         - Определиться с parent
        """
        action_params = {'fake_id': action['fake_id']} if action.get('fake_id') else {'id': action['id']}
        for key, val in action.items():
            if isinstance(val, dict) and DepartmentSection.has_value(key):
                action_params.update(val)
        if action.get('delete'):
            action_params['delete'] = True

        if action_params.get('chief'):
            action_params['chief'] = (
                Staff.objects.get(id=action_params['chief'])
            )

        if 'deputies' in action_params:
            action_params['deputies'] = (
                Staff.objects.filter(id__in=action_params['deputies'])
            )

        if DepartmentSection.HIERARCHY.value in action:
            parent = self._fake_id_to_dep.get(action_params.pop('fake_parent', None))
            action_params['parent'] = parent or Department.objects.filter(id=action_params.get('parent')).first()

        for fname in ('clubs', 'maillists'):
            if fname in action_params:
                action_params[fname] = [list(d.values())[0] for d in action_params[fname]]

        if action_params.get('kind'):
            action_params['kind'] = DepartmentKind.objects.get(id=action_params.get('kind'))

        return action_params

    def create_department(self, params):
        new_department_ctl = department.DepartmentCtl.create(
            parent=params.get('parent'),
            code=params.pop('code'),
            kind_id=params.pop('kind').id,
            author_user=self._proposal_ctl.author.user,
            oebs_structure_date=self._proposal_ctl.effective_date,
        )
        fake_id = params['fake_id']
        for field, value in params.items():
            setattr(new_department_ctl, field, value)
        new_department_ctl.save()

        self._fake_id_to_dep[fake_id] = new_department_ctl.instance

    def edit_department(self, department_id, edit_params):
        delete_department = edit_params.pop('delete', False)
        department_ctl = department.DepartmentCtl(
            department=department_id,
            author_user=self._proposal_ctl.author.user,
        )
        for param_name, param_value in edit_params.items():
            setattr(department_ctl, param_name, param_value)
        department_ctl.save()

        if delete_department:
            department_ctl.delete()

    def _execute_person_department_action(self, action):
        if 'department' not in action:
            return

        params = {'department': self._get_dep_for_person_action(action['department'])}
        action['department']['department'] = params['department'].url

        controller = self._login_to_person_ctl[action['login']]
        controller.update(params)
        controller.save(self._proposal_ctl.author)

    def _get_dep_for_person_action(self, dep_params):
        fake_id = dep_params.get('fake_department')
        if fake_id:
            return self._fake_id_to_dep[fake_id]

        return self._url_to_dep[dep_params['department']]

    def _execute_person_action(self, action):
        params = {}
        position_section = action.get(PersonSection.POSITION.value)
        if position_section:
            params['position'] = position_section['new_position']
        organization_section = action.get(PersonSection.ORGANIZATION.value)
        if organization_section:
            params['organization'] = self._id_to_org[organization_section['organization']]
        office_section = action.get(PersonSection.OFFICE.value)
        if office_section:
            params['office'] = self._id_to_office[office_section['office']]

        if self.intern_transfer_detector.is_intern_transfer_to_staff(action):
            params['date_completion_internship'] = None

        controller = self._login_to_person_ctl[action['login']]
        controller.update(params)
        controller.save(self._proposal_ctl.author)

    def _execute_vacancy_action(self, action: Action):
        should_execute_action = action.get('department') or action.get('fake_department')
        if not should_execute_action:
            return

        params = {'department': self._get_dep_for_vacancy_or_headcount_action(action)}
        action['department'] = params['department'].url

        controller = self._vacancy_id_to_vacancy_ctl[action['vacancy_id']]
        controller.update(params)
        controller.save(self._proposal_ctl.author)

    def _execute_headcount_action(self, action: Action):
        should_execute_action = action.get('department') or action.get('fake_department')
        if not should_execute_action:
            return

        params = {'department': self._get_dep_for_vacancy_or_headcount_action(action)}
        action['department'] = params['department'].url

        controller = self._headcount_code_to_headcount_ctl[action['headcount_code']]
        controller.update(params)
        controller.save(self._proposal_ctl.author)

    def _get_dep_for_vacancy_or_headcount_action(self, action_params):
        fake_id = action_params.get('fake_department')
        if fake_id:
            return self._fake_id_to_dep[fake_id]

        return self._url_to_dep[action_params['department']]

    def create_proposal_for_splitted_actions(self):
        splitted_person_actions = self._proposal_ctl.proposal_object['persons'].get('splitted_actions', [])
        splitted_department_actions = self._proposal_ctl.proposal_object.get('splitted_actions', [])

        if not (splitted_person_actions or splitted_department_actions):
            return

        for action in splitted_person_actions:
            if action.get('department', {}).get('fake_department'):
                action['department'].update({
                    'department': self._fake_id_to_dep[action['department']['fake_department']].url,
                    'fake_department': '',
                })

        new_proposal_params = {
            'apply_at': self._proposal_ctl.apply_at,
            'apply_at_hr': self._proposal_ctl.proposal_object['apply_at_hr'],
            'description': self._proposal_ctl.proposal_object['description'],
            'actions': splitted_department_actions,
            'persons': {
                'actions': splitted_person_actions,
            },
            'tickets': {
                'persons': self._proposal_ctl.proposal_object['tickets'].get('splitted_persons', {}),
                'deleted_persons': self._proposal_ctl.proposal_object['tickets'].get('deleted_persons', {}),
            },
        }

        self._proposal_ctl.update_workflow_changes()

        proposal_ctl = ProposalCtl(author=self._proposal_ctl.author).create_from_cleaned_data(new_proposal_params)
        proposal_ctl.splitted_from = self.proposal_id
        self._proposal_ctl.splitted_to = proposal_ctl.save()

    def _set_metadata_applied(self, execution_author_login):
        meta = ProposalMetadata.objects.get(proposal_id=self.proposal_id)
        meta.applied_at = time_now()
        if execution_author_login:
            meta.applied_by = Staff.objects.get(login=execution_author_login)
        else:
            meta.applied_by = self._proposal_ctl.author
        meta.save()

    def _confirm_workflows_on_execute(self):
        WorkflowRegistryUpdater(self._proposal_ctl).confirm_changes_for_proposal()

    def actualize_affiliations(self):
        """Если в заявке есть перемещение между корневыми
        подразделениями - актуализируем affiliation сотрудников
        Может быть выполнено только после применения заявки
        """
        if not self._proposal_ctl.applied_at():
            return

        if len(self._proposal_ctl.proposal_object['root_departments']) == 1:
            return

        parents = {}
        all_parents = set()

        for action in self._proposal_ctl.department_actions:
            if not is_moving(action):
                continue
            hierarchy = action[DepartmentSection.HIERARCHY.value]

            old_state = self._proposal_ctl.proposal_object['old_state']
            old_parent = old_state[str(action['id'])][DepartmentSection.HIERARCHY.value]['parent']
            new_parent = hierarchy['parent'] or self._fake_id_to_dep[hierarchy['fake_parent']].id

            parents[action['id']] = (old_parent, new_parent)
            all_parents |= {old_parent, new_parent}

        tree_ids = {
            dep_id: tree_id
            for dep_id, tree_id in Department.objects.filter(id__in=all_parents).values_list('id', 'tree_id')
        }

        for dep_id, (old_parent, new_parent) in parents.items():
            if tree_ids.get(old_parent) != tree_ids.get(new_parent):
                actualize_affiliation_for_department(dep_id)


def actualize_affiliation_for_department(department_id: int):
    logger.info('Actualizing affiliation for department %s', department_id)
    department = Department.objects.filter(intranet_status=1).get(id=department_id)
    persons_qs = Staff.objects.filter(
        is_dismissed=False,
        department__tree_id=department.tree_id,
        department__lft__gte=department.lft,
        department__rght__lte=department.rght,
    )
    for person in persons_qs:
        actualize_affiliation(person)
        person.save()
