from itertools import chain
from typing import Any, Dict, Iterator, List, Optional, Union, Tuple

import attr
from staff.lib import waffle
from django.core.cache import cache
from django.utils.functional import SimpleLazyObject

from staff.lib.models.departments_chain import get_departments_tree
from staff.lib.models.roles_chain import get_roles_by_departments_qs

from staff.person.models.person import Staff
from staff.proposal.models import DepartmentAttrs

from staff.departments.controllers.tickets.helpers.table_flow import TableFlowAPI
from staff.departments.models import Department, DepartmentRoles


DEPARTMENT_ATTRS_TTL = 5 * 60  # 5 min
DEPARTMENT_ATTRS_PREFIX = 'DEP_ATTRS_%s'


class DepartmentAttrsCtl:

    def __init__(self):
        self._attrs = None

    def __call__(self, login: str, department_ids: Iterator[int]) -> Dict[str, Optional[Union[str, List[str]]]]:
        budget_found = False

        result = {
            'analyst': None,
            'budget_owner': None,
            'budget_tags': [],
            'budget_notify': [],
        }

        for dep_id in chain(department_ids, [None, 0]):
            if waffle.switch_is_active('enable_table_flow'):
                attrs = cache.get(DEPARTMENT_ATTRS_PREFIX % dep_id)
                if not attrs:
                    attrs = TableFlowAPI.get_department_attrs(dep_id)
                    cache.set(DEPARTMENT_ATTRS_PREFIX % dep_id, attrs, DEPARTMENT_ATTRS_TTL)
            else:
                attrs = self.attrs.get(dep_id)

            if not attrs:
                continue

            if attrs['budget_tag']:
                result['budget_tags'].append(attrs['budget_tag'])

            owner = attrs['budget_owner']
            if not budget_found and owner and owner != login:
                budget_found = True
                self.set_budget(result, attrs, login)

            if not result['analyst']:
                result['analyst'] = (
                    None if attrs['analyst'] == login
                    else attrs['analyst']
                )

        if len(result['budget_tags']) > 1:
            result['budget_tags'] = result['budget_tags'][:-1]

        return result

    @classmethod
    def set_budget(cls, result, attrs, login):
        result['budget_owner'] = attrs['budget_owner']

        result['budget_notify'] = list(set(attrs['budget_notify']))
        if login in result['budget_notify']:
            result['budget_notify'] = []

    @property
    def attrs(self) -> Dict[int, Dict[str, Union[str, List]]]:
        if self._attrs is None:
            budget_notify_qs = (
                DepartmentAttrs.budget_notify.through.objects
                .values(
                    'departmentattrs',
                    'staff__login',
                )
            )

            budget_notify: Dict[int, Union[str, List]] = {}
            for notify in budget_notify_qs:
                budget_notify.setdefault(
                    notify['departmentattrs'], []
                ).append(notify['staff__login'])

            attrs = (
                DepartmentAttrs.objects
                .values(
                    'id',
                    'department_id',
                    'analyst__login',
                    'budget_owner__login',
                    'budget_tag',
                )
            )

            self._attrs = {
                a['department_id']: {
                    'analyst': a['analyst__login'],
                    'budget_owner': a['budget_owner__login'],
                    'budget_tag': a['budget_tag'],
                    'budget_notify': budget_notify.get(a['id'], []),
                }
                for a in attrs
            }
        return self._attrs


class DepartmentAttrsBaseGetter:
    """
    Лениво кеширует в себя все DepartmentAttrs
    По id подразделения отдаёт его DepartmentAttrs (.get_one)
        либо список DepartmentAttrs всей его цепочки (.get_all)
    """

    cache_one: Dict[int, Any] = {}
    cache_all: Dict[int, List[Any]] = {}

    # Полный набор DepartmentAttrs. Меняется редко.
    cached_attrs: Dict[int, Any] = None
    cached_default: Any = None

    def _fetch_all_department_attrs(self) -> None:
        raise NotImplementedError

    @property
    def attrs(self) -> Dict[int, Any]:
        if not self.cached_attrs:
            self._fetch_all_department_attrs()
        return self.cached_attrs

    @property
    def default(self) -> Any:
        if not self.cached_default:
            self.cached_default = self.attrs.get(0) or self.attrs.get(None)
        return self.cached_default

    def get_analyst(self, department: Department):
        first_role = get_roles_by_departments_qs(
            [department],
            (DepartmentRoles.HR_ANALYST_TEMP.value,),
            ('login',)
        ).first()
        return first_role and first_role['staff__login']

    def get_one(self, dep_id: int) -> Optional[Any]:
        """По id подразделения вычисляет его аттрибут"""
        if dep_id in self.attrs:
            return self.attrs[dep_id]

        if dep_id in self.cache_one:
            return self.cache_one[dep_id]

        all_attrs = self.get_all(dep_id)
        self.cache_one[dep_id] = all_attrs[0] if all_attrs else None
        return self.cache_one[dep_id]

    def get_all(self, dep_id: int) -> List[Any]:
        """Cписок аттрибутов всей цепочки по id подразделения"""
        if dep_id in self.cache_all:
            return self.cache_all[dep_id]

        dep_chain = get_departments_tree([dep_id], fields=['id'])[dep_id]

        all_attrs = []
        for dep in reversed(dep_chain):
            if dep['id'] in self.attrs:
                all_attrs.append(self.attrs[dep['id']])

        self.cache_all[dep_id] = all_attrs
        return all_attrs


class DepartmentAttrsDBGetter(DepartmentAttrsBaseGetter):

    def _fetch_all_department_attrs(self) -> None:
        attrs_qs = (
            DepartmentAttrs.objects
            .select_related('analyst', 'department')
            .prefetch_related('budget_notify', 'ticket_access')
        )
        self.cached_attrs = {attr.department_id: attr for attr in attrs_qs}


@attr.s(auto_attribs=True, frozen=True)
class Attrs:
    department_id: int = attr.ib(default=None)
    analyst: Staff = attr.ib(default=None, converter=lambda login: Staff.objects.filter(login=login).first())
    budget_owner: Staff = attr.ib(default=None, converter=lambda login: Staff.objects.filter(login=login).first())
    budget_tag: str = attr.ib(factory=str)
    budget_notify: Tuple[str, ...] = attr.ib(factory=tuple)
    ticket_access: Tuple[str, ...] = attr.ib(factory=tuple)


class DepartmentAttrsTableFlowGetter(DepartmentAttrsBaseGetter):

    def _fetch_all_department_attrs(self) -> None:
        attrs_list = [
            Attrs(**TableFlowAPI.get_department_attrs(dep_id))
            for dep_id in TableFlowAPI.get_all_department_attrs_ids()
        ]
        self.cached_attrs = {attr.department_id: attr for attr in attrs_list}


def department_attrs_getter():
    if waffle.switch_is_active('enable_table_flow'):
        return DepartmentAttrsTableFlowGetter()
    else:
        return DepartmentAttrsDBGetter()


# Настройки DepartmentAttrs меняются крайне редко
department_attrs = SimpleLazyObject(department_attrs_getter)
