import logging
from typing import Any, Dict, List, Optional, Union

import requests
from staff.lib import waffle
from django.conf import settings
from requests.exceptions import ConnectionError, Timeout


logger = logging.getLogger(__name__)


LIST_FIELDS = (
    'budget_notify',
    'ticket_access',
    'all_department_attrs',
)

SPECIAL_FIELDS = (
    'all_department_attrs',
)


class TableFlowAPI:

    @classmethod
    def _request(cls, method: str, url: str, params: Dict[str, Union[str, int]] = None) -> Dict[str, str]:
        headers = {
            'Authorization': 'OAuth %s' % settings.ROBOT_STAFF_OAUTH_TOKEN,
        }

        try:
            response = requests.request(
                method=method,
                url=f'https://{settings.TABLE_FLOW_HOST}{url}',
                params=params,
                headers=headers,
                timeout=10,
            )
        except (ConnectionError, Timeout):
            logger.exception('Table-flow is not responding')
            return {}

        if not response.ok:
            logger.warning(
                'Table-flow error %s, response: %s',
                response.status_code,
                response.content.decode('utf-8'),
            )
            return {}

        try:
            return response.json() if response.content else {}
        except ValueError:
            logger.exception(
                'Error during parse Table-flow response: %s',
                response.content.decode('utf-8'),
            )
            return {}

    @classmethod
    def _get(cls, url: str, params: Dict[str, Union[str, int]] = None) -> Dict[str, str]:
        return cls._request('GET', url, params=params)

    @classmethod
    def get_department_attrs(cls, department_id: int) -> Dict[str, Any]:
        if not waffle.switch_is_active('enable_table_flow'):
            return {}

        url = '/rules/staff_dep_attrs/execute/'
        response = cls._get(url, params={'department_id': department_id})
        if not response:
            return {}

        to_tuple = []
        for k, v in response.items():
            if k in LIST_FIELDS:
                to_tuple.append(k)

            if v == 'None':
                response[k] = None

        for k in to_tuple:
            if response[k]:
                raw_list = response[k].split(',')
                if all(i.isdigit() for i in raw_list):
                    response[k] = tuple(int(i) for i in raw_list)
                else:
                    response[k] = tuple(raw_list)

        for k in SPECIAL_FIELDS:
            response.pop(k, None)

        response['department_id'] = department_id
        return response

    @classmethod
    def get_all_department_attrs_ids(cls) -> List[int]:
        if not waffle.switch_is_active('enable_table_flow'):
            return []

        url = '/rules/staff_dep_attrs/execute/'
        response = cls._get(url, params={'department_id': 0})
        if not response:
            return []

        return [int(i) for i in response['all_department_attrs'].split(',')]

    @classmethod
    def get_all_department_attrs(cls) -> List[Dict[str, Any]]:
        if not waffle.switch_is_active('enable_table_flow'):
            return []

        return [
            cls.get_department_attrs(dep_id)
            for dep_id in cls.get_all_department_attrs_ids()
        ]

    @classmethod
    def get_base_analyst_login(cls) -> Optional[str]:
        if not waffle.switch_is_active('enable_table_flow'):
            return None

        url = '/rules/staff_dep_attrs/execute/'
        response = cls._get(url, params={'department_id': 0})
        if not response:
            return None

        return response['analyst']
