import logging
import re
from copy import deepcopy
from typing import Any, Dict, List

from staff.lib import waffle
from django.conf import settings
from django.template import Context, loader
from django.utils import translation
from ids.exceptions import EmptyIteratorError
from ids.services.startrek2.connector import ConflictMonster

from staff.lib.startrek.issues import create_issue, get_issue

from staff.groups.models import Group, GROUP_TYPE_CHOICES, GroupMembership
from staff.payment.enums import WAGE_SYSTEM

from staff.departments.controllers.proposal_action import PersonSection
from staff.departments.controllers.tickets import helpers
from staff.departments.controllers.tickets.base import (
    ProposalContext,
    ProposalTicketCtlError,
)


logger = logging.getLogger(__name__)

wage_translations = {
    WAGE_SYSTEM.PIECEWORK: {'ru': 'Сдельная', 'en': 'Piecework'},
    WAGE_SYSTEM.HOURLY: {'ru': 'Почасовая', 'en': 'Hourly'},
    WAGE_SYSTEM.FIXED: {'ru': 'Фиксированная', 'en': 'Fixed'},
}


class IssueType:
    POSITION = 'position'
    MONEY = 'money'
    MOVING = 'moving'
    MOVING_WITH_MONEY = 'moneyMoving'


class PersonTicket:

    TICKET_TEMPLATE_RU = 'startrek/person/ru/base.html'
    TICKET_TEMPLATE_EN = 'startrek/person/en/base.html'

    COMMENT_TEMPLATE = 'startrek/person_comment.html'
    FROZEN_FIELDS = ('access', 'followers', 'assignee')
    NOTIFY_AUTHOR_ON_UPDATING = True

    def __init__(self, proposal_context: ProposalContext, person_login: str):
        self.context = proposal_context
        self.login = person_login
        self.person = proposal_context.persons_data[person_login]['instance']

        self.ticket_key = self._get_person_ticket_key()
        self._issue = None

        self.action: Dict[str, Any] = self.get_action()
        self.person_data: Dict[str, Any] = proposal_context.persons_data[person_login]
        self.dep_data: Dict[str, Any] = proposal_context.dep_data[self.person_data['department_id']]

        self._ticket_context: Dict = {}
        self._ensure_deleted_person_key()

    @classmethod
    def from_proposal_id(cls, proposal_id: str, person_login: str):
        context = ProposalContext.from_proposal_id(proposal_id)
        return cls(context, person_login)

    @classmethod
    def from_proposal_ctl(cls, proposal_ctl, person_login: str):
        context = ProposalContext(proposal_ctl)
        return cls(context, person_login)

    def get_action(self) -> Dict[str, Any]:
        for person_action in self.context.person_actions:
            if person_action['login'] == self.login:
                return person_action

    @property
    def issue(self):
        if self._issue is None:
            try:
                self._issue = get_issue(unique=self.get_unique())
            except EmptyIteratorError:
                logger.warning('Error trying to fetch ticket by filter `%s`', f'unique:{self.get_unique()}')
            if self._issue is None:
                try:
                    ticket_key = self.context.proposal_object['tickets']['persons'].get(self.login)
                    self._issue = ticket_key and get_issue(key=ticket_key)
                except EmptyIteratorError:
                    logger.warning('Error trying to fetch ticket by key `%s`', f'key:{ticket_key}')
            if self._issue is None:
                raise ProposalTicketCtlError(f'Proposal {self.context.proposal_id} has no ticket')
        return self._issue

    def _get_person_ticket_key(self) -> str:
        return (
            self.context.proposal_object['tickets']['persons'].get(self.login, '')
            or self.context.proposal_object['tickets']['deleted_persons'].get(self.login, '')
        )

    def _has_active_person_ticket(self) -> bool:
        return bool(
            self.context.proposal_object['tickets']['persons'].get(self.login, '')
        )

    def _set_department_data_to_context(self, ticket_context: Dict[str, Any]) -> None:
        get_department_attrs = helpers.DepartmentAttrsCtl()
        ticket_context['old_department'] = self.context.dep_chains[self.person_data['department_id']]
        ticket_context['old_department']['service_groups'] = [
            gm.group for gm in GroupMembership.objects
            .filter(
                group__type=GROUP_TYPE_CHOICES.SERVICE,
                group__intranet_status=1,
                staff__login=ticket_context['login'],
            )
        ]
        ticket_context['old_department_attrs'] = get_department_attrs(
            self.login,
            reversed(ticket_context['old_department']['ids']),
        )

        ticket_context['department_attrs'] = ticket_context['old_department_attrs']
        department_section = ticket_context.get(PersonSection.DEPARTMENT.value)
        if department_section:
            new_dep_url = department_section['department']
            if new_dep_url:
                new_dep = self.context.dep_data[new_dep_url]['instance']
                department_section['department'] = new_dep
                department_section.update(self.context.dep_chains[new_dep.id])
            else:
                fake_dep = department_section['fake_department']
                department_section['fake_department'] = self.context.dep_data[fake_dep]
                department_section['dep_ticket'] = self.context.proposal_object['tickets']['department_ticket']
                department_section['restructurisation_ticket'] = (
                    self.context.proposal_object['tickets']['restructurisation']
                )

            ticket_context['department_attrs'] = get_department_attrs(
                self.login,
                reversed(ticket_context['department'].get('ids', [])),
            )

            department_section['service_groups'] = list(
                Group.objects.filter(url__in=department_section['service_groups'])
            )

    def _set_office_data_to_context(self, ticket_context: Dict[str, Any]) -> None:
        office_section = ticket_context.get(PersonSection.OFFICE.value)
        if office_section:
            ticket_context[PersonSection.OFFICE.value] = {
                'office': self.context.offices[office_section['office']],
                'need_food_compensation': False,
            }

    def _set_position_data_to_context(self, ticket_context: Dict[str, Any]) -> None:
        position_section = ticket_context.get(PersonSection.POSITION.value)
        if position_section:
            job_id = position_section['position_legal']
            job_object = self.context.positions[job_id]
            position_section['position'] = {'ru': job_object.name, 'en': job_object.name_en}

    def _set_grade_data_to_context(self, ticket_context: Dict[str, Any]) -> None:
        grade_section = ticket_context.get(PersonSection.GRADE.value)
        if grade_section:
            grade_section['new_grade'] = self.context.grades[
                grade_section['new_grade']
            ]
            new_occupation = self.action[PersonSection.GRADE.value].get('occupation')
            if new_occupation:
                grade_section['new_occupation'] = self.context.grades

    def _set_organization_data_to_context(self, ticket_context: Dict[str, Any]) -> None:
        organization_section = ticket_context.get(PersonSection.ORGANIZATION.value)
        if organization_section:
            org_id = organization_section['organization']
            ticket_context[PersonSection.ORGANIZATION.value] = {
                'organization': self.context.organizations[org_id]
            }

    def _set_salary_data_to_context(self, ticket_context: Dict[str, Any]) -> None:
        salary_section = ticket_context.get(PersonSection.SALARY.value)
        if salary_section:
            for ws in ('old_wage_system', 'new_wage_system'):
                salary_section[ws] = wage_translations[salary_section[ws]]

    def _set_value_stream_data_to_context(self, ticket_context: Dict[str, Any]) -> None:
        value_stream_section = ticket_context.get(PersonSection.VALUE_STREAM.value)
        if value_stream_section:
            ticket_context['hr_product'] = self.context.hr_product_data.get(value_stream_section.get('value_stream'))

    def _set_geography_data_to_context(self, ticket_context: Dict[str, Any]) -> None:
        geography_section = ticket_context.get(PersonSection.GEOGRAPHY.value)
        if geography_section:
            ticket_context['geography'] = self.context.geography_data.get(geography_section.get('geography'))

    @property
    def ticket_context(self):
        if self._ticket_context:
            return self._ticket_context

        tc = deepcopy(self.action)
        tc['person'] = self.person_data['instance']

        self._set_department_data_to_context(tc)
        self._set_office_data_to_context(tc)
        self._set_position_data_to_context(tc)
        self._set_grade_data_to_context(tc)
        self._set_organization_data_to_context(tc)
        self._set_salary_data_to_context(tc)
        self._set_value_stream_data_to_context(tc)
        self._set_geography_data_to_context(tc)

        get_city_attrs = helpers.CityAttrsCtl()
        tc['city_attrs'] = get_city_attrs(
            tc['person'],
            tc[PersonSection.OFFICE.value]['office'] if PersonSection.OFFICE.value in tc else None,
        )
        author_is_chief = helpers.AuthorIsChief(self.context.proposal.author)
        tc['author_is_chief'] = author_is_chief(tc['person'])
        tc['is_author'] = tc['person'].id == self.context.proposal_object['author']

        if tc['author_is_chief']:
            official_positions = self.context.official_positions
            tc['old_position'] = official_positions.get(tc['person'].id)

        tc['current_staff_position'] = {'ru': tc['person'].position, 'en': tc['person'].position_en}

        tc['comment'] = {'comment': tc['comment']}
        self._ticket_context = tc

        return self._ticket_context

    def update_ticket(self, author_login: str) -> str:
        """
        Обновляет все поля тикета кроме `FROZEN_FIELDS` соответственно текущему состоянию заявки
        Оставляет комментарий про редактирование заявки
        """
        ticket_params = self.generate_all_ticket_params()
        for field_name in self.FROZEN_FIELDS:
            ticket_params.pop(field_name, None)

        ticket_params.pop('queue', None)
        ticket_params.pop('type', None)
        ticket_params.pop('version', None)
        ticket_params.pop('unique', None)

        for _ in range(3):
            try:
                self.issue.update(
                    params={'notifyAuthor': self.NOTIFY_AUTHOR_ON_UPDATING},
                    **ticket_params
                )
                break
            except ConflictMonster:
                self._issue = None
                logger.warning('Failed to save changes issue. Exception: ConflictMonster. ticket: %s', self.ticket_key)

        self.add_comment(
            comment_text='Тикет был обновлен кем:%s при редактировании заявки:'
            '\n<{Новое состояние:\n%s \n}>' % (author_login, ticket_params['description'])
        )
        return self.ticket_key

    def create_ticket(self, author_login: str = None, adding_to_existing_proposal: bool = False) -> str:
        """
        Создаёт тикет по сотруднику, либо обновляет, если он был создан ранее и удалён в последствии
        Пересохраняет заявку, добавив в неё ключ тикета
        Возвращает строку с ключом тикета
        """
        author_login = author_login or self.context.proposal.author.login
        ticket_params = self.generate_all_ticket_params()
        self._issue = create_issue(**ticket_params)
        self.ticket_key = self.issue.key

        if adding_to_existing_proposal:
            self.add_comment(f'Сотрудник был добавлен в заявку кем:{author_login}')

        self.context.proposal.update_tickets(person_tickets={self.login: self.ticket_key})
        self.context.proposal.save()
        return self.ticket_key

    def renew_ticket(self, author_login: str = None) -> str:
        """
        обновляет тикет по сотруднику если он был создан ранее и удалён в последствии
        Пересохраняет заявку, добавив в неё ключ тикета
        Возвращает строку с ключом тикета
        """
        author_login = author_login or self.context.proposal.author.login
        deleted_persons: Dict[str, str] = self.context.proposal_object['tickets'].get('deleted_persons', {})
        if self.login not in deleted_persons:
            raise ValueError(
                f'Error trying to renew ticket of {self.login} while it\'s not in `deleted_persons`. '
                f'Proposal: {self.context.proposal_id}'
            )

        self.context.proposal.update_tickets(person_tickets={self.login: self.ticket_key})
        self.context.proposal.save()
        self.update_ticket(author_login=author_login)
        self.add_comment(f'Сотрудник был снова добавлен в заявку кем:{author_login}')
        return self.ticket_key

    def delete_from_proposal(self, author_login: str, comment_text: str = '') -> None:
        """
        Пока что ничего с полями тикета не делает, но может делать, если понадобится
        Оставляет в тикете коммент про удаление сотрудника из заявки
        """
        self.context.proposal.update_tickets(deleted_person_tickets={self.login: self.ticket_key})
        self.context.proposal.save()
        self.add_comment(
            comment_text
            or f'Сотрудник {self.login} был удален из заявки кем:{author_login}'
        )
        self.ticket_key = ''

    def move_to_r15n(self, author_login: str, r15n_ticket_key: str) -> None:
        """
        Удаляем так же из `persons` и переносим в `deleted_persons`, но с комментарием
        что согласование переехало в тикет реструктуризации
        """
        return self.delete_from_proposal(
            author_login=author_login,
            comment_text=f'Согласование было перемещено в тикет рестуктуризации: {r15n_ticket_key}',
        )

    def add_comment_if_has_ticket(self, comment_text: str):
        if self._has_active_person_ticket():
            self.add_comment(comment_text)

    def add_comment(self, comment_text: str) -> None:
        """Постит в тикет коммент от имени Общего Блага"""
        try:
            self.issue.comments.create(text=comment_text.strip())
        except Exception as e:
            logger.error(
                'Error trying to add comment to ticket %s. Comment text: %s. Exception: %s',
                self.ticket_key,
                comment_text,
                e,
            )
            raise e

    def generate_all_ticket_params(self) -> Dict[str, Any]:
        ticket_description = self.get_description()
        personal_ticket_params = self.generate_personal_ticket_params()
        return {
            'queue': settings.PROPOSAL_QUEUE_ID,
            'type': self.get_ticket_type(),
            'summary': self.get_summary(),
            'assignee': self.get_assignee(),
            'createdBy': self.context.proposal.author.login,
            'description': ticket_description,
            'followers': self.get_followers(),
            'access': self.get_accessors(),
            'tags': self.get_tags(),
            'components': self.get_components(),
            'fixVersions': self.get_versions(),

            'unique': self.get_unique(),
            **personal_ticket_params,
        }

    def generate_personal_ticket_params(self) -> Dict[str, Any]:
        params = {
            'raiseDate': self.context.proposal_object['apply_at'].isoformat(),

            'toCreate': '; '.join(self.context.get_creating_department_names()),
            'toMove': '; '.join(self.context.get_moving_department_names()),
            'toDelete': '; '.join(self.context.get_deleting_department_names()),

            'employees': [self.login],
            'department': self.ticket_context['old_department']['direction']['chain_ru'],
            'currentDepartment': self.ticket_context['old_department']['chain_ru'],
            'hr': self.ticket_context['city_attrs']['hr'],
        }

        fields_to_reset = [
            'allHead', 'currentHead', 'newHead', 'analytics', 'analyst', 'budgetHolder',
            'reason', 'currentSalary', 'newSalary', 'salarySystem', 'totalLevelChanges',
            'newPosition', 'newDepartment', 'newOffice', 'legalEntity2',
        ]
        params.update({field: None for field in fields_to_reset})  # обнуляем поля тикета при изменении заявки

        old_department = self.ticket_context['old_department']
        old_chiefs = [c for c in old_department['chiefs'] if c != self.login]
        if old_chiefs:
            params['allHead'] = old_chiefs
            params['currentHead'] = old_chiefs[-1]

        analytics = self.get_analytics()
        if analytics:
            params['analytics'] = analytics

        analyst = self.ticket_context['department_attrs']['analyst']
        if analyst:
            params['analyst'] = analyst

        budget_owner = self.ticket_context['department_attrs']['budget_owner']
        if budget_owner:
            params['budgetHolder'] = budget_owner

        if PersonSection.SALARY.value in self.ticket_context:
            salary = self.ticket_context['salary']
            old_salary_t = '{old_salary} {old_currency} ({old_rate})'
            new_salary_t = '{new_salary} {new_currency} ({new_rate})'
            params['reason'] = self.ticket_context['comment']['comment']
            params['currentSalary'] = old_salary_t.format(**salary)
            params['newSalary'] = new_salary_t.format(**salary)
            if salary['old_wage_system'] != salary['new_wage_system']:
                params['salarySystem'] = salary['new_wage_system']['ru']

        if PersonSection.GRADE.value in self.ticket_context:
            params['totalLevelChanges'] = self.ticket_context['grade']['new_grade']['level']

        position_section = self.ticket_context.get(PersonSection.POSITION.value)
        if position_section:
            params['newPosition'] = position_section['position']['ru']

        new_department_data = self.ticket_context.get(PersonSection.DEPARTMENT.value)
        if new_department_data:
            new_dep_chain = new_department_data.get('chain_ru')
            if new_dep_chain:
                params['newDepartment'] = new_dep_chain
            else:
                new_fake_dep_name = new_department_data.get('fake_department', {}).get('ru')
                params['newDepartment'] = new_fake_dep_name

            new_chiefs = [c for c in new_department_data.get('chiefs', []) if c != self.login]
            if new_chiefs:
                old_chiefs = params.get('allHead') or []
                params['allHead'] = list({*new_chiefs, *old_chiefs})
                params['newHead'] = new_chiefs[-1]

        office_section = self.ticket_context.get(PersonSection.OFFICE.value)
        if office_section and not waffle.switch_is_active('disable_office_filling_for_proposal'):
            params['newOffice'] = office_section['office'].id

        organization_section = self.ticket_context.get(PersonSection.ORGANIZATION.value)
        if organization_section and not waffle.switch_is_active('disable_org_filling_for_proposal'):
            params['legalEntity2'] = organization_section['organization']['st_translation_id']

        return params

    def get_description(self) -> str:
        template_ru = loader.get_template(self.TICKET_TEMPLATE_RU)
        template_en = loader.get_template(self.TICKET_TEMPLATE_EN)

        context = self.ticket_context.copy()
        context['settings'] = settings
        context['proposal_id'] = self.context.proposal_id
        context['blocks'] = list(self.ticket_context.keys())
        context['base'] = {
            'date': self.context.proposal_object['apply_at'],
            'comment': self.context.proposal_object['description'],
        }

        context = Context(context)

        with translation.override('ru'):
            description_ru = template_ru.render(context)
        with translation.override('en'):
            description_en = template_en.render(context)

        description = '\n'.join([description_ru, description_en])
        description = re.sub(r'\n{3,}', '\n\n', description).strip()

        return description

    def get_ticket_type(self) -> str:
        """
        Тип 1. Изменение только должности – position
        Тип 2. Изменение зарплаты без перевода – money OR (money AND position)
        Тип 3. Перевод без изменения зарплаты – moving OR (moving AND position)
        Тип 4. Перевод c изменением зарплаты –
            (money AND moving) OR (money AND position AND moving)
        """
        money = PersonSection.SALARY.value in self.ticket_context
        changing_office = PersonSection.OFFICE.value in self.ticket_context
        changing_department = PersonSection.DEPARTMENT.value in self.ticket_context
        moving = changing_office or changing_department

        if money and moving:
            return IssueType.MOVING_WITH_MONEY
        elif moving:
            return IssueType.MOVING
        elif money:
            return IssueType.MONEY
        else:
            return IssueType.POSITION

    def get_summary(self) -> str:
        blocks = {
            'salary': 'зарплаты',
            'position': 'должности',
            'grade': 'уровня должности',
            'department': 'подразделения',
            'office': 'офиса',
            'organization': 'организации',
        }
        blocks_row = ', '.join([blocks[n] for n in blocks if n in self.action])
        blocks_row = re.sub(r', ([^,]+)$', r' и \1', blocks_row)

        first_name = self.person_data['first_name']
        last_name = self.person_data['last_name']
        login = self.person_data['login']

        if self.action.get('department', {}).get('from_maternity_leave'):
            return f'Выход из декретного отпуска: {first_name} {last_name}, {login}@'
        return f'Изменение {blocks_row}: {first_name} {last_name}, {login}@'

    def get_assignee(self) -> str:
        return self.ticket_context['old_department_attrs']['analyst']

    def get_followers(self) -> List[str]:
        return self.ticket_context['department_attrs']['budget_notify'][:]

    def get_accessors(self) -> List[str]:
        budget_holders: set = {
            self.ticket_context['department_attrs']['budget_owner'],
            self.ticket_context['old_department_attrs']['budget_owner'],
        }
        budget_holders |= set(self.ticket_context['department_attrs']['budget_notify'])
        budget_holders |= set(self.ticket_context['old_department_attrs']['budget_notify'])

        budget_holders.discard(None)

        return list(budget_holders)

    def get_tags(self) -> List[str]:
        tags = []

        new_department_data = self.ticket_context.get(PersonSection.DEPARTMENT.value)
        if new_department_data:
            if new_department_data['from_maternity_leave']:
                tags.append('декрет')
            if not new_department_data['with_budget']:
                tags.append('ротация')

        return tags

    def get_components(self):
        if waffle.switch_is_active('disable_proposal_st_components'):
            return []

        new_budget_tags = self.ticket_context['department_attrs']['budget_tags']
        old_budget_tags = self.ticket_context['old_department_attrs']['budget_tags']

        components = list({*old_budget_tags, *new_budget_tags})
        return components

    def get_versions(self):
        city_name = self.ticket_context['city_attrs']['name']
        if city_name:
            return [city_name]
        return []

    def get_analytics(self) -> List[str]:
        old_analyst = self.ticket_context['old_department_attrs']['analyst']
        new_anallyst = self.ticket_context['department_attrs']['analyst']
        return [analyst for analyst in (old_analyst, new_anallyst) if analyst]

    def get_unique(self):
        return f'proposal/{self.context.proposal_id}/{self.login}'

    def _ensure_deleted_person_key(self) -> None:
        """В старых заявках ключ deleted_persons мог отсутствовать, мы тогда не хранили их. Проставляем его."""
        proposal = self.context.proposal
        if 'deleted_persons' not in proposal.proposal_object['tickets']:
            proposal.proposal_object['tickets']['deleted_persons'] = {}

    def __repr__(self):
        return f'<PersonTicket `{self.ticket_key}` ({self.login}) for {self.context.proposal_id}>'
