import logging
from typing import Any, Dict, List, Set

from django.conf import settings

from staff.departments.controllers.proposal_action import Action
from staff.departments.controllers.tickets import RestructurisationTicket, helpers
from staff.departments.controllers.tickets.base import ProposalTicketCtlError


logger = logging.getLogger(__name__)


# TODO: надо выделить общего предка, с хедкаунт тикетом тож самое
class ValueStreamTicket(RestructurisationTicket):
    FROZEN_FIELDS = ('assignee', )
    TICKET_TEMPLATE = 'startrek/value_stream_ticket.html'
    queue_id = settings.HEADCOUNT_QUEUE_ID

    def get_ticket_key(self) -> str:
        return self.context.proposal_object['tickets'].get('value_stream')

    def generate_ticket_context(self, enumerate_entries=True):
        total = len(
            list(self.ticket_dispatcher.value_stream_vacancy_actions) +
            list(self.ticket_dispatcher.value_stream_headcount_actions) +
            list(self.ticket_dispatcher.value_stream_person_actions)
        )
        return {
            'proposal_id': self.context.proposal_id,
            'settings': settings,
            'description': self.context.proposal_object['description'],
            'apply_at': self.get_apply_date(),
            'vacancies': [
                self.prepare_vacancy_action_ticket_context(action)
                for action in self.ticket_dispatcher.value_stream_vacancy_actions
            ],
            'headcounts': [
                self.prepare_headcount_action_ticket_context(action)
                for action in self.ticket_dispatcher.value_stream_headcount_actions
            ],
            'persons': [
                self.prepare_person_action_ticket_context(action)
                for action in self.ticket_dispatcher.value_stream_person_actions
            ],
            'total': total,
        }

    def prepare_vacancy_action_ticket_context(self, action_data: Action) -> Dict[str, Any]:
        result = {
            'new_hr_product': None,
            'new_hr_product_head': None,
            'old_hr_product': None,
            'old_hr_product_head': None,
            'new_geography': None,
            'old_geography': None,
            'department': None,
            'department_head': None,
            'vacancy_id': None,
            'name': None,
            'headcount_position_code': None,
            'ticket': None,
            'status': None,
            'comment': None,
        }
        vacancy_data = self.context.vacancies_data[action_data['vacancy_id']]
        hc_code = vacancy_data['headcount_position_code']
        old_value_stream = self.context.existing_value_streams.get(hc_code)
        if old_value_stream:
            old_hr_product = self.context.hr_products[old_value_stream]
            result['old_hr_product'] = old_hr_product.hr_product_slug
            result['old_hr_product_head'] = old_hr_product.hr_product_head

        value_stream = action_data.get('value_stream')
        if value_stream:
            new_hr_product = self.context.hr_products[value_stream]
            result['new_hr_product'] = new_hr_product.hr_product_slug
            result['new_hr_product_head'] = new_hr_product.hr_product_head

        result['department'] = {'url': vacancy_data['department__url']}
        result['vacancy_id'] = action_data['vacancy_id']
        result['name'] = vacancy_data['name']
        result['headcount_position_code'] = vacancy_data['headcount_position_code']
        result['ticket'] = vacancy_data['ticket']
        result['status'] = vacancy_data['status']
        result['comment'] = action_data.get('comment', '')

        self._put_geography_to_context(result, hc_code, action_data)

        return result

    def prepare_person_action_ticket_context(self, action_data: Action) -> Dict[str, Any]:
        result = {
            'new_hr_product': None,
            'new_hr_product_head': None,
            'old_hr_product': None,
            'old_hr_product_head': None,
            'new_geography': None,
            'old_geography': None,
            'department': None,
            'department_head': None,
            'login': None,
            'comment': None,
        }
        login = action_data['login']
        person_data = self.context.persons_data[login]
        hc_code = self.context.person_budget_positions.get(login)

        if hc_code:
            old_value_stream = self.context.existing_value_streams.get(hc_code)
            if old_value_stream:
                old_hr_product = self.context.hr_products[old_value_stream]
                result['old_hr_product'] = old_hr_product.hr_product_slug
                result['old_hr_product_head'] = old_hr_product.hr_product_head

            value_stream = action_data.get('value_stream', {}).get('value_stream')
            if value_stream:
                new_hr_product = self.context.hr_products[value_stream]
                result['new_hr_product'] = new_hr_product.hr_product_slug
                result['new_hr_product_head'] = new_hr_product.hr_product_head

        result['department'] = {'url': person_data['department__url']}
        result['login'] = login
        result['comment'] = action_data.get('comment', '')

        self._put_geography_to_context(result, hc_code, action_data.get('geography', {}))

        return result

    def prepare_headcount_action_ticket_context(self, action_data: Action) -> Dict[str, Any]:
        result = {
            'new_hr_product': None,
            'new_hr_product_head': None,
            'old_hr_product': None,
            'old_hr_product_head': None,
            'new_geography': None,
            'old_geography': None,
            'department': None,
            'department_head': None,
            'code': None,
            'name': None,
            'status': None,
            'comment': None,
        }
        code = action_data['headcount_code']
        headcount_data = self.context.headcounts_data[code]

        old_value_stream = self.context.existing_value_streams.get(code)
        if old_value_stream:
            old_hr_product = self.context.hr_products[old_value_stream]
            result['old_hr_product'] = old_hr_product.hr_product_slug
            result['old_hr_product_head'] = old_hr_product.hr_product_head

        value_stream = action_data.get('value_stream')
        if value_stream:
            new_hr_product = self.context.hr_products[value_stream]
            result['new_hr_product'] = new_hr_product.hr_product_slug
            result['new_hr_product_head'] = new_hr_product.hr_product_head

        result['code'] = code
        result['name'] = headcount_data['name']
        result['status'] = headcount_data['status']
        result['department'] = {'url': headcount_data['department__url']}
        result['comment'] = action_data.get('comment', '')

        self._put_geography_to_context(result, code, action_data)

        return result

    def delete_ticket(self, author_login: str, comment_text: str) -> None:
        """
        Пока что просто пишет комментарий в тикет и удаляет ключ тикета из заявки
        """
        self.add_comment(author_login, comment_text=comment_text)
        tickets = self.context.proposal_object['tickets']
        tickets['deleted_value_stream'] = tickets['value_stream']
        self.ticket_key = ''
        self.update_ticket_key_in_proposal()

    def create_ticket(self, author_login=None) -> str:
        """
        Создаёт тикет в стартреке в очереди HEADCOUNT (THEADCOUNT для тестинга).
        Примеры ключей в ticket_params: summary, description, assignee, tags, followers
        Пересохраняет заявку, добавив в неё ключ тикета
        Возвращает строку с ключом тикета
        """
        if self.ticket_key:
            raise ProposalTicketCtlError(self.ticket_key)
        old_key = self.context.proposal_object['tickets'].get('deleted_value_stream')
        if old_key:
            self._restore_ticket(old_key, author_login)
        else:
            self._create_ticket()
        self.update_ticket_key_in_proposal()
        return self.ticket_key

    def update_ticket_key_in_proposal(self):
        self.context.proposal.update_tickets(value_steam_ticket=self.ticket_key)
        self.context.proposal.save()

    def get_unique(self):
        return f'proposal/{self.context.proposal_id}/value_stream'

    def get_summary(self):
        return 'Заявка на изменение изменение HR-продукта (VS) или географии задач'

    def get_accessors(self) -> List[str]:
        accessors = {
            product.hr_product_head
            for product in self.context.hr_products.values() if product.hr_product_head
        }
        current_accessors = set()

        if self.ticket_key:
            current_accessors = {x.login for x in self.issue.access}
            logger.info('Current accessors %s', current_accessors)

        return list(accessors.union(current_accessors))

    def get_assignee(self):
        if len(self.analysts) > 0:
            return next(iter(self.analysts))

        return helpers.department_attrs.default.analyst.login

    def _get_headcount_analysts(self) -> Set[str]:
        return {
           helpers.department_attrs.get_analyst(
               self.context.get_department(
                   self.context.headcounts_data[act['headcount_code']]['department_id']
               )
           )
           for act in self.context.headcount_actions
        } - {None}
