from datetime import datetime

from django.contrib.auth.models import AnonymousUser

from staff.audit.factory import create_log
from staff.person.models import Staff

from staff.departments.models import Vacancy


class VacancyCtl(object):
    IMMUTABLE_FIELDS = {
        'pk', 'id', 'created_at'
    }
    ALL_FIELDS = {f.name for f in Vacancy._meta.fields}
    EDITABLE_FIELDS = ALL_FIELDS - IMMUTABLE_FIELDS

    def __init__(self, vacancy):
        self.instance = vacancy
        self.CTL_FIELDS = [
            f for f in self.ALL_FIELDS if f in VacancyCtl.__dict__
        ]

    @property
    def old_state(self):
        if not hasattr(self, '_old_state'):
            if self.instance.id:
                self._old_state = Vacancy.objects.get(id=self.instance.id)
            else:
                self._old_state = None
        return self._old_state

    def __getattr__(self, name):
        try:
            return super(VacancyCtl, self).__getattr__(name)
        except AttributeError:
            if name in self.CTL_FIELDS:
                raise
            return getattr(self.instance, name)

    def __setattr__(self, name, value):

        is_vacancy_attr = (
            name != 'instance'
            and hasattr(self.instance, name)
            and name not in self.CTL_FIELDS
        )

        if is_vacancy_attr:
            setattr(self.instance, name, value)
        else:
            super(VacancyCtl, self).__setattr__(name, value)

    def save(self, author_user=None):
        self.modified_at = datetime.now()
        self.old_state  # форсирую получение старого состояния до сохранения
        self.instance.save()

        if isinstance(author_user, Staff):
            author_user = author_user.user
        if not (author_user is None or isinstance(author_user, AnonymousUser)):
            create_log(
                objects=[self.instance],
                who=author_user,
                action='vacancy_updated',
                primary_key=self.id
            )
        return self

    def update(self, data):
        self._preprofile_id = data.pop('preprofile_id', None)
        for attr, value in data.items():
            if attr in self.EDITABLE_FIELDS or attr == 'preprofile_id':
                setattr(self, attr, value)
        return self
