from datetime import datetime
from typing import Any, Dict

from staff.groups.models import Group, GROUP_TYPE_CHOICES
from staff.person.models import Staff

from staff.departments.models import Department, DepartmentStaff, ValuestreamRoles, HeadcountPosition


def remove_value_streams_roles():
    DepartmentStaff.objects.filter(role_id=ValuestreamRoles.HEAD.value).delete()


def remove_value_streams():
    HeadcountPosition.objects.filter(valuestream__isnull=False).update(valuestream=None)
    Department.valuestreams.all().delete()


def create_value_streams_services():
    now = datetime.now()
    default_params = {
        'intranet_status': 1,
        'position': 0,
        'created_at': now,
        'modified_at': now,
        'service_tags': 'oebs_use_for_hr',
        'type': GROUP_TYPE_CHOICES.SERVICE,
    }

    def create_group(name, code, service_id: int, parent: Group or None = None) -> Dict[str, Any]:
        url = f'svc_{code}'
        params = {'parent': parent, 'name': name, 'code': code, 'url': url, 'service_id': service_id, **default_params}
        return Group.objects.update_or_create(url=url, defaults=params)[0]

    meta_search = create_group('Сервисы поискового портала', 'meta_search', 851)
    alice = create_group('Алиса (VS)', 'voice', 675, parent=meta_search)
    create_group('Умные устройства', 'vsdev', 1455, parent=alice)
    create_group('Реклама', 'ad', 31735, parent=meta_search)
    create_group('Внутренние и b2b сервисы', 'tools', 872, parent=meta_search)
    create_group('Услуги', 'ydo', 2776, parent=meta_search)
    create_group('Маркетинг', 'vsmarketing', 32986, parent=meta_search)
    create_group('Поиск', 'vs_search', 31752, parent=meta_search)
    create_group('Безопасность', 'security', 909, parent=meta_search)

    create_group('Руководство', 'topmanagement', 2904)
    create_group('Геоинформационные сервисы', 'meta_content', 863)

    taxi = create_group('Сервисы Такси', 'taxi', 435)
    create_group('Яндекс.Драйв', 'yandexcarsharing', 1796, parent=taxi)
    create_group('Еда', 'eda', 2228, parent=taxi)
    create_group('Яндекс.Лавка', 'yandexlavka', 4041, parent=taxi)

    cloud = create_group('Яндекс.Облако', 'cloud', 1415)
    create_group('Яндекс.Облако (Бизнес)', 'ycb', 4331, parent=cloud)


def _get_staff_id(login: str) -> int:
    return Staff.objects.get(login=login).id


def _get_group_service_id(url: str) -> int:
    return Group.objects.get(url=url, type=GROUP_TYPE_CHOICES.SERVICE).service_id


def get_services_heads(role_id):
    from staff.departments.controllers.value_streams_rollup import ValueStreamRoleId

    if role_id == ValueStreamRoleId.HEAD:
        yield _get_staff_id('styskin'), _get_group_service_id('svc_meta_search')
        yield _get_staff_id('shuster'), _get_group_service_id('svc_voice')
        yield _get_staff_id('rnefyodov'), _get_group_service_id('svc_vsdev')
        yield _get_staff_id('dkachmar'), _get_group_service_id('svc_ad')
        yield _get_staff_id('veged'), _get_group_service_id('svc_tools')
        yield _get_staff_id('mborisov'), _get_group_service_id('svc_ydo')
        yield _get_staff_id('abroskin'), _get_group_service_id('svc_vsmarketing')
        yield _get_staff_id('zagrebin'), _get_group_service_id('svc_vs_search')
        yield _get_staff_id('tokza'), _get_group_service_id('svc_security')
        yield _get_staff_id('bunina'), _get_group_service_id('svc_topmanagement')
        yield _get_staff_id('rommich'), _get_group_service_id('svc_meta_content')
        yield _get_staff_id('anikinan1989'), _get_group_service_id('svc_taxi')
        yield _get_staff_id('aryazanov'), _get_group_service_id('svc_yandexcarsharing')
        yield _get_staff_id('dmasyuk'), _get_group_service_id('svc_eda')
        yield _get_staff_id('krasil'), _get_group_service_id('svc_yandexlavka')
        yield _get_staff_id('abash'), _get_group_service_id('svc_cloud')
    else:
        return
